@extends('layouts.admin')

@section('content')
    <div class="bg-white rounded-xl shadow-lg p-6">
        {{-- Header with Action Buttons --}}
        <div
            class="flex flex-col md:flex-row justify-between items-start md:items-center mb-6 gap-4 border-b border-gray-100 pb-6">
            <h2 class="text-2xl font-bold bg-gradient-to-r from-emerald-600 to-teal-600 bg-clip-text text-transparent">
                {{ __('Client Details') }} - {{ $client->arabic_name }}
            </h2>

            <div class="flex flex-wrap gap-2">
                <a href="{{ route('clients.index') }}"
                    class="inline-flex items-center gap-1 px-3 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600 transition duration-150 shadow-sm">
                    <i class="entypo-back"></i>
                    <span>{{ __('Back') }}</span>
                </a>

                <a href="{{ route('clients.edit', $client->cl_id) }}"
                    class="inline-flex items-center gap-1 px-3 py-2 bg-amber-500 text-white rounded-lg hover:bg-amber-600 transition duration-150 shadow-sm">
                    <i class="entypo-pencil"></i>
                    <span>{{ __('Edit') }}</span>
                </a>

                <button onclick="openReceiptsModal()"
                    class="inline-flex items-center gap-1 px-3 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition duration-150 shadow-sm">
                    <i class="entypo-doc-text"></i>
                    <span>{{ __('View Receipts') }}</span>
                </button>

                <button onclick="openReservationsModal()"
                    class="inline-flex items-center gap-1 px-3 py-2 bg-purple-500 text-white rounded-lg hover:bg-purple-600 transition duration-150 shadow-sm">
                    <i class="entypo-home"></i>
                    <span>{{ __('Reservations') }}</span>
                </button>

                <button onclick="openCheckModal()"
                    class="inline-flex items-center gap-1 px-3 py-2 bg-teal-500 text-white rounded-lg hover:bg-teal-600 transition duration-150 shadow-sm">
                    <i class="entypo-doc-text-inv"></i>
                    <span>{{ __('Print Check') }}</span>
                </button>

                @if($cnt > 0)
                    <a href="{{ route('clients.live_data', $client->member_no) }}" target="_blank"
                        class="inline-flex items-center gap-1 px-3 py-2 bg-info text-white rounded-lg hover:bg-info/90 transition duration-150 shadow-sm">
                        <i class="entypo-pencil"></i>
                        <span>{{ __('Live Data') }}</span>
                    </a>
                @endif

                {{-- Old Data button hidden as requested --}}

                @if($client->rev == 'N')
                    <button onclick="toggleReviewStatus('Y')"
                        class="inline-flex items-center gap-1 px-3 py-2 bg-emerald-500 text-white rounded-lg hover:bg-emerald-600 transition duration-150 shadow-sm">
                        <i class="entypo-check"></i>
                        <span>{{ __('Mark Reviewed') }}</span>
                    </button>
                @else
                    <button onclick="toggleReviewStatus('N')"
                        class="inline-flex items-center gap-1 px-3 py-2 bg-orange-500 text-white rounded-lg hover:bg-orange-600 transition duration-150 shadow-sm">
                        <i class="entypo-ccw"></i>
                        <span>{{ __('Re-review') }}</span>
                    </button>
                @endif
            </div>
        </div>

        {{-- Review Status Indicator --}}
        <div class="mb-4">
            <div id="reviewStatus"
                class="inline-block px-4 py-2 rounded-lg border-2 {{ $client->rev == 'Y' ? 'border-emerald-500 bg-emerald-50 text-emerald-700' : 'border-red-500 bg-red-50 text-red-700' }} font-semibold">
                {{ $client->rev == 'Y' ? __('Reviewed') : __('Not Reviewed') }}
            </div>
        </div>

        {{-- Client Information Table --}}
        <div class="overflow-x-auto rounded-xl border border-gray-200">
            <table class="min-w-full divide-y divide-gray-200">
                <tbody class="bg-white divide-y divide-gray-100">
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50 w-1/5">
                            {{ __('Arabic Name') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->arabic_name }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-100">{{ __('Parking Due') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('English Name') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->english_name }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-100">
                            {{ __('Parking Paid') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('National ID') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->nat_id }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-100">
                            {{ __('Parking Arrears') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">
                            {{ __('Military Card') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->milit_card_id }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-orange-100">{{ __('Roof Due') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Address') }}</th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->address }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-orange-100">{{ __('Roof Paid') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Nationality') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->nationality }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-orange-100">
                            {{ __('Roof Arrears') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900 bg-emerald-50">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Mobile') }}</th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->mobile }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-orange-100">{{ __('Gas Paid') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Gender') }}</th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->gender }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-emerald-100">
                            {{ __('Garden Size') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('City') }}</th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->city }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-emerald-100">
                            {{ __('Garden Due') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('District') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->state }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-emerald-100">
                            {{ __('Garden Paid') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">
                            {{ __('Personal Email') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->per_email }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-emerald-100">
                            {{ __('Garden Arrears') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">0</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Teba Email') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->teba_email }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Printed By') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ auth()->user()->name ?? 'Admin' }}</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Username') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->user_name }}</td>
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Print Time') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ now()->format('d-M-Y h:i:s') }}</td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Bank Name') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->bank_name }}</td>
                        <td colspan="2"></td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">
                            {{ __('Account Number') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->account_no }}</td>
                        <td colspan="2"></td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('File Number') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->file_no }}</td>
                        <td colspan="2"></td>
                    </tr>
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">
                            {{ __('Member Number') }}
                        </th>
                        <td class="px-4 py-3 text-sm text-gray-900">{{ $client->old_member_no }}</td>
                        <td colspan="2"></td>
                    </tr>
                    @if($reservation)
                        <tr class="hover:bg-gray-50">
                            <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Project') }}</th>
                            <td class="px-4 py-3 text-sm text-gray-900">{{ $reservation->project_name ?? '-' }}</td>
                            <td colspan="2"></td>
                        </tr>
                        <tr class="hover:bg-gray-50">
                            <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Area') }}</th>
                            <td class="px-4 py-3 text-sm text-gray-900">{{ $reservation->flat_txt ?? '-' }} -
                                {{ $sessArr[$reservation->sess ?? '1'] ?? '' }} ({{ $subsess }})
                            </td>
                            <td colspan="2"></td>
                        </tr>
                        <tr class="hover:bg-gray-50">
                            <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Assignment') }}
                            </th>
                            <td class="px-4 py-3 text-sm text-gray-900">{{ $reservation->bldg_id ?? '-' }} --
                                {{ $reservation->flat_num ?? '-' }}
                            </td>
                            <td colspan="2"></td>
                        </tr>
                    @endif
                    <tr class="hover:bg-gray-50">
                        <th class="px-4 py-3 text-start text-sm font-bold text-gray-700 bg-gray-50">{{ __('Comments') }}
                        </th>
                        <td colspan="3" class="px-4 py-3 text-sm text-red-600 font-semibold">{{ $client->comment }}</td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>

    {{-- Receipts Modal --}}
    <div id="receiptsModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 overflow-y-auto">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-2xl shadow-2xl w-full max-w-7xl max-h-[90vh] overflow-hidden">
                <div class="bg-gradient-to-r from-blue-600 to-blue-700 px-6 py-4 flex justify-between items-center">
                    <h3 class="text-xl font-bold text-white">{{ __('Client Receipts') }}</h3>
                    <button onclick="closeReceiptsModal()" class="text-white hover:text-gray-200">
                        <i class="entypo-cancel text-2xl"></i>
                    </button>
                </div>

                <div class="p-6 overflow-y-auto max-h-[calc(90vh-200px)]">
                    <div class="overflow-x-auto">
                        <table id="receiptsTable" class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Serial') }}
                                    </th>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Receipt No') }}
                                    </th>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Bank No') }}
                                    </th>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Value') }}
                                    </th>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Date') }}
                                    </th>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Type') }}
                                    </th>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Entry Date') }}
                                    </th>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Account') }}
                                    </th>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Project') }}
                                    </th>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Code') }}
                                    </th>
                                    <th class="px-3 py-3 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Actions') }}
                                    </th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-100">
                                @php $serial = 1; @endphp
                                @foreach($receipts as $receipt)
                                    <tr class="hover:bg-gray-50 {{ $receipt->rev == 'Y' ? 'bg-yellow-50' : '' }}">
                                        <td class="px-3 py-2 text-center text-sm">{{ $serial++ }}</td>
                                        <td class="px-3 py-2 text-center text-sm">{{ $receipt->number }}</td>
                                        <td class="px-3 py-2 text-center text-sm">{{ $receipt->bnk_number }}</td>
                                        <td class="px-3 py-2 text-center text-sm font-semibold">
                                            {{ number_format($receipt->value) }}
                                        </td>
                                        <td class="px-3 py-2 text-center text-sm">{{ $receipt->date }}</td>
                                        <td class="px-3 py-2 text-center text-sm">{{ $receipt->type_txt ?? '-' }}</td>
                                        <td class="px-3 py-2 text-center text-sm">{{ $receipt->created_at }}</td>
                                        <td class="px-3 py-2 text-center text-sm">{{ $receipt->bank_name ?? '-' }}</td>
                                        <td class="px-3 py-2 text-center text-sm">{{ $receipt->prj_id }}</td>
                                        <td class="px-3 py-2 text-center text-sm">{{ $receipt->chk == 'Y' ? 'شيك' : 'ايصال' }}
                                        </td>
                                        <td class="px-3 py-2 text-center">
                                            <div class="flex gap-1 justify-center">
                                                <button onclick="editReceipt({{ $receipt->id }})"
                                                    class="inline-flex items-center gap-1 px-2 py-1 bg-amber-500 text-white rounded text-xs hover:bg-amber-600"
                                                    title="{{ __('Edit') }}">
                                                    <i class="entypo-pencil"></i>
                                                    <span>{{ __('Edit') }}</span>
                                                </button>
                                                <button onclick="deleteReceipt({{ $receipt->id }})"
                                                    class="inline-flex items-center gap-1 px-2 py-1 bg-red-500 text-white rounded text-xs hover:bg-red-600"
                                                    title="{{ __('Delete') }}">
                                                    <i class="entypo-trash"></i>
                                                    <span>{{ __('Delete') }}</span>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                @endforeach
                                <tr class="bg-gray-100 font-bold">
                                    <td colspan="3" class="px-3 py-3 text-center">{{ __('Total') }}</td>
                                    <td colspan="8" class="px-3 py-3 text-center">{{ number_format($total) }}</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>

                <div class="bg-gray-50 px-6 py-4 flex justify-between items-center border-t">
                    <button onclick="closeReceiptsModal()"
                        class="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600">
                        {{ __('Close') }}
                    </button>
                    <button onclick="openAddReceiptModal()"
                        class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                        {{ __('Add Receipt') }}
                    </button>
                </div>
            </div>
        </div>
    </div>

    {{-- Add Receipt Modal --}}
    <div id="addReceiptModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 overflow-y-auto">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-2xl shadow-2xl w-full max-w-2xl">
                <div class="bg-gradient-to-r from-emerald-600 to-emerald-700 px-6 py-4 flex justify-between items-center">
                    <h3 class="text-xl font-bold text-white">{{ __('Add Receipt') }}</h3>
                    <button onclick="closeAddReceiptModal()" class="text-white hover:text-gray-200">
                        <i class="entypo-cancel text-2xl"></i>
                    </button>
                </div>

                <form id="addReceiptForm" class="p-6">
                    @csrf
                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Receipt Number') }}</label>
                            <input type="text" name="number" id="receipt_number"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500"
                                required>
                        </div>
                        <div>
                            <label
                                class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Bank Receipt Number') }}</label>
                            <input type="text" name="bnk_number" id="bnk_receipt_number"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500"
                                required>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Value') }}</label>
                            <input type="number" name="value"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500"
                                required>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Date') }}</label>
                            <input type="text" name="date"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500"
                                placeholder="dd-mm-yyyy" required>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Project') }}</label>
                            <select name="prj_id"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500"
                                required>
                                @foreach($projects as $project)
                                    <option value="{{ $project->prj_id }}">{{ $project->name }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Type') }}</label>
                            <select name="type"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500"
                                required>
                                @foreach($receiptTypes as $type)
                                    <option value="{{ $type->id }}">{{ $type->type }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="col-span-2">
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Bank Account') }}</label>
                            <select name="bank_account"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500"
                                required>
                                @foreach($banks as $bank)
                                    <option value="{{ $bank->bank_id }}">{{ $bank->bank_name }} - {{ $bank->account_number }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>

                    <div class="mt-6 flex justify-end gap-3">
                        <button type="button" onclick="closeAddReceiptModal()"
                            class="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600">
                            {{ __('Cancel') }}
                        </button>
                        <button type="submit" class="px-4 py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700">
                            {{ __('Save') }}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    {{-- Edit Receipt Modal --}}
    <div id="editReceiptModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 overflow-y-auto">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-2xl shadow-2xl w-full max-w-2xl">
                <div class="bg-gradient-to-r from-amber-600 to-amber-700 px-6 py-4 flex justify-between items-center">
                    <h3 class="text-xl font-bold text-white">{{ __('Edit Receipt') }}</h3>
                    <button onclick="closeEditReceiptModal()" class="text-white hover:text-gray-200">
                        <i class="entypo-cancel text-2xl"></i>
                    </button>
                </div>

                <form id="editReceiptForm" class="p-6">
                    @csrf
                    <input type="hidden" id="edit_receipt_id" name="receipt_id">

                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Receipt Number') }}</label>
                            <input type="text" name="number" id="edit_receipt_number"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500"
                                required>
                        </div>
                        <div>
                            <label
                                class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Bank Receipt Number') }}</label>
                            <input type="text" name="bnk_number" id="edit_bnk_receipt_number"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500"
                                required>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Value') }}</label>
                            <input type="number" name="value" id="edit_value"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500"
                                required>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Date') }}</label>
                            <input type="text" name="date" id="edit_date"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500"
                                placeholder="dd-mm-yyyy" required>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Project') }}</label>
                            <select name="prj_id" id="edit_prj_id"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500"
                                required>
                                @foreach($projects as $project)
                                    <option value="{{ $project->prj_id }}">{{ $project->name }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Type') }}</label>
                            <select name="type" id="edit_type"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500"
                                required>
                                @foreach($receiptTypes as $type)
                                    <option value="{{ $type->id }}">{{ $type->type }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="col-span-2">
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Bank Account') }}</label>
                            <select name="bank_account" id="edit_bank_account"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500"
                                required>
                                @foreach($banks as $bank)
                                    <option value="{{ $bank->bank_id }}">{{ $bank->bank_name }} - {{ $bank->account_number }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>

                    <div class="mt-6 flex justify-end gap-3">
                        <button type="button" onclick="closeEditReceiptModal()"
                            class="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600">
                            {{ __('Cancel') }}
                        </button>
                        <button type="submit" class="px-4 py-2 bg-amber-600 text-white rounded-lg hover:bg-amber-700">
                            {{ __('Update') }}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    {{-- Reservations Modal --}}
    <div id="reservationsModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 overflow-y-auto">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-2xl shadow-2xl w-full max-w-4xl max-h-[90vh] overflow-hidden">
                <div class="bg-gradient-to-r from-purple-600 to-purple-700 px-6 py-4 flex justify-between items-center">
                    <h3 class="text-xl font-bold text-white">{{ __('Client Reservations') }}</h3>
                    <button onclick="closeReservationsModal()" class="text-white hover:text-gray-200">
                        <i class="entypo-cancel text-2xl"></i>
                    </button>
                </div>

                <div class="p-6 overflow-y-auto max-h-[calc(90vh-100px)]">
                    {{-- Status Messages --}}
                    <div id="resvMsg" class="hidden mb-4 p-3 rounded-lg text-white font-medium"></div>

                    {{-- Existing Reservations --}}
                    <h4 class="text-lg font-bold text-gray-800 mb-3 border-b pb-2">{{ __('Current Reservations') }}</h4>
                    <div class="overflow-x-auto mb-6">
                        <table class="min-w-full divide-y divide-gray-200 border">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-3 py-2 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Project') }}
                                    </th>
                                    <th class="px-3 py-2 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Building') }}
                                    </th>
                                    <th class="px-3 py-2 text-center text-xs font-bold text-gray-700 uppercase">
                                        {{ __('Floor') }}
                                    </th>
                                    <th class="px-3 py-2 text-center text-xs font-semibold text-gray-600">{{ __('Flat') }}
                                    </th>
                                    <th class="px-3 py-2 text-center text-xs font-semibold text-gray-600">{{ __('Date') }}
                                    </th>
                                    <th class="px-3 py-2 text-center text-xs font-semibold text-gray-600">
                                        {{ __('Actions') }}
                                    </th>
                                </tr>
                            </thead>
                            <tbody id="reservationsList" class="bg-white divide-y divide-gray-100">
                                {{-- Loaded via AJAX --}}
                            </tbody>
                        </table>
                    </div>

                    {{-- Add Reservation Form --}}
                    <h4 class="text-lg font-bold text-gray-800 mb-3 border-b pb-2">{{ __('Add New Reservation') }}</h4>
                    <form id="addReservationForm" class="bg-gray-50 p-4 rounded-xl border border-gray-200">
                        @csrf
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Project') }}</label>
                                <select name="prj_id" id="resv_prj_id" onchange="loadBuildings()"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500"
                                    required>
                                    <option value="">{{ __('Select Project') }}</option>
                                    @foreach($projects as $project)
                                        <option value="{{ $project->prj_id }}">{{ $project->name }}</option>
                                    @endforeach
                                </select>
                            </div>
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Building') }}</label>
                                <input type="text" name="bldg_id" id="resv_bldg_id"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500"
                                    required>
                            </div>
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Floor') }}</label>
                                <input type="text" name="floor" id="resv_floor"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500"
                                    required>
                            </div>
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Flat') }}</label>
                                <input type="text" name="flat_num" id="resv_flat_num"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500"
                                    required>
                            </div>
                        </div>
                        <div class="mt-4 flex justify-end gap-3">
                            <button type="button" onclick="closeReservationsModal()"
                                class="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600 transition">
                                {{ __('Close') }}
                            </button>
                            <button type="submit"
                                class="px-6 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition">
                                <i class="entypo-plus"></i> {{ __('Add Reservation') }}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    {{-- Withdrawal (Check) Modal --}}
    <div id="checkModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 overflow-y-auto">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-2xl shadow-2xl w-full max-w-lg">
                <div class="bg-gradient-to-r from-teal-600 to-teal-700 px-6 py-4 flex justify-between items-center">
                    <h3 class="text-xl font-bold text-white">{{ __('Print Check') }}</h3>
                    <button onclick="closeCheckModal()" class="text-white hover:text-gray-200">
                        <i class="entypo-cancel text-2xl"></i>
                    </button>
                </div>

                <form action="{{ route('clients.print_check') }}" method="POST" target="_blank" class="p-6">
                    @csrf
                    <div class="space-y-4">
                        <div>
                            <label
                                class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Pay to the order of') }}</label>
                            <input type="text" name="name"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-teal-500"
                                required>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Value') }}</label>
                            <input type="number" name="value" step="0.01"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-teal-500"
                                required>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Date') }}</label>
                            <input type="text" name="date"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-teal-500"
                                value="{{ date('d-m-Y') }}" required>
                        </div>
                        <div>
                            <label class="block text-sm font-semibold text-gray-700 mb-1">{{ __('Bank') }}</label>
                            <select name="bank_id"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-teal-500"
                                required>
                                @foreach($banks as $bank)
                                    <option value="{{ $bank->bank_id }}">{{ $bank->bank_name }}</option>
                                @endforeach
                            </select>
                        </div>
                    </div>

                    <div class="mt-6 flex justify-end gap-3">
                        <button type="button" onclick="closeCheckModal()"
                            class="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600">
                            {{ __('Cancel') }}
                        </button>
                        <button type="submit" class="px-4 py-2 bg-teal-600 text-white rounded-lg hover:bg-teal-700">
                            <i class="entypo-print"></i> {{ __('Print') }}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        const clientId = {{ $client->cl_id }};

        function openCheckModal() {
            document.getElementById('checkModal').classList.remove('hidden');
        }

        function closeCheckModal() {
            document.getElementById('checkModal').classList.add('hidden');
        }

        function openReceiptsModal() {
            document.getElementById('receiptsModal').classList.remove('hidden');
        }

        function closeReceiptsModal() {
            document.getElementById('receiptsModal').classList.add('hidden');
        }

        function openAddReceiptModal() {
            document.getElementById('addReceiptModal').classList.remove('hidden');
        }

        function closeAddReceiptModal() {
            document.getElementById('addReceiptModal').classList.add('hidden');
        }

        function openReservationsModal() {
            document.getElementById('reservationsModal').classList.remove('hidden');
            loadReservations();
        }

        function closeReservationsModal() {
            document.getElementById('reservationsModal').classList.add('hidden');
        }

        function loadReservations() {
            fetch(`{{ route('clients.reservations.get', $client->cl_id) }}`)
                .then(res => res.json())
                .then(data => {
                    const list = document.getElementById('reservationsList');
                    list.innerHTML = '';
                    if (data.reservations.length > 0) {
                        data.reservations.forEach(res => {
                            list.innerHTML += `
                                                                <tr class="hover:bg-gray-50">
                                                                    <td class="px-3 py-2 text-center text-sm">${res.project_name || '-'}</td>
                                                                    <td class="px-3 py-2 text-center text-sm">${res.bldg_id}</td>
                                                                    <td class="px-3 py-2 text-center text-sm">${res.floor}</td>
                                                                    <td class="px-3 py-2 text-center text-sm">${res.flat_txt || res.flat_num}</td>
                                                                    <td class="px-3 py-2 text-center text-sm">${res.resv_date}</td>
                                                                    <td class="px-3 py-2 text-center text-sm">
                                                                        <button onclick="deleteReservation(${res.prj_id}, ${res.bldg_id}, '${res.flat_num}')" class="text-red-500 hover:text-red-700 font-bold py-1 px-2 rounded">
                                                                        <i class="fa fa-trash"></i> {{ __('Delete') }}
                                                                    </button>
                                                                    </td>
                                                                </tr>
                                                            `;
                        });
                    } else {
                        list.innerHTML = '<tr><td colspan="6" class="px-4 py-4 text-center text-gray-500">{{ __("No reservations found") }}</td></tr>';
                    }
                });

        }

        function deleteReservation(prj_id, bldg_id, flat_num) {
            if (confirm('{{ __("Are you sure you want to delete this reservation?") }}')) {
                fetch(`{{ route('clients.reservations.delete') }}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json',
                        'X-Requested-With': 'XMLHttpRequest',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({
                        prj_id: prj_id,
                        bldg_id: bldg_id,
                        flat_num: flat_num,
                        _method: 'DELETE'
                    })
                })
                    .then(response => {
                        if (!response.ok) {
                            return response.text().then(text => {
                                throw new Error(text || 'Server error');
                            });
                        }
                        return response.json();
                    })
                    .then(data => {
                        if (data.success) {
                            loadReservations();
                            alert('{{ __("Reservation deleted successfully") }}');
                        } else {
                            alert(data.error || '{{ __("Error deleting reservation") }}');
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        alert('{{ __("An error occurred") }}: ' + error.message);
                    });
            }
        }


        // Auto-calculate floor based on flat number
        document.getElementById('resv_flat_num').addEventListener('input', function () {
            const flatNum = parseInt(this.value);
            const floorInput = document.getElementById('resv_floor');

            if (isNaN(flatNum)) {
                floorInput.value = '';
                return;
            }

            if (flatNum < 100) {
                floorInput.value = 'الارضى';
            } else {
                // < 200 -> 1, < 300 -> 2, etc.
                floorInput.value = Math.floor(flatNum / 100);
            }
        });

        // Add Reservation Form Submit
        document.getElementById('addReservationForm').addEventListener('submit', function (e) {
            e.preventDefault();
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);

            fetch(`{{ route('clients.reservations.add', $client->cl_id) }}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify(data)
            })
                .then(response => {
                    if (!response.ok) {
                        return response.text().then(text => {
                            throw new Error(text || 'Server error');
                        });
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        alert('{{ __("Reservation added successfully") }}');
                        loadReservations();
                        // Reset form
                        this.reset();
                    } else if (data.error) {
                        alert(data.error);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    // Try to parse error message if it's JSON
                    try {
                        const errObj = JSON.parse(error.message);
                        if (errObj.message) {
                            alert(errObj.message);
                        } else if (errObj.error) {
                            alert(errObj.error);
                        } else if (errObj.errors) {
                            // Show validation errors
                            let msg = '';
                            for (const [key, value] of Object.entries(errObj.errors)) {
                                msg += value + '\n';
                            }
                            alert(msg);
                        } else {
                            alert('{{ __("An error occurred") }}');
                        }
                    } catch (e) {
                        alert('{{ __("An error occurred") }}: ' + error.message);
                    }
                });
        });

        function toggleReviewStatus(status) {
            if (confirm('{{ __("Are you sure?") }}')) {
                fetch(`{{ route('clients.review_status', $client->cl_id) }}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({ status: status })
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            location.reload();
                        }
                    });
            }
        }

        function openEditReceiptModal() {
            document.getElementById('editReceiptModal').classList.remove('hidden');
        }

        function closeEditReceiptModal() {
            document.getElementById('editReceiptModal').classList.add('hidden');
        }

        function editReceipt(id) {
            // Fetch receipt data
            fetch(`{{ route('clients.receipts.get_data') }}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({ id: id, cl_id: clientId })
            })
                .then(response => response.json())
                .then(data => {
                    if (data.receipt) {
                        const receipt = data.receipt;

                        // Populate form fields
                        document.getElementById('edit_receipt_id').value = receipt.id;
                        document.getElementById('edit_receipt_number').value = receipt.number;
                        document.getElementById('edit_bnk_receipt_number').value = receipt.bnk_number;
                        document.getElementById('edit_value').value = receipt.value;
                        document.getElementById('edit_date').value = receipt.date;
                        document.getElementById('edit_prj_id').value = receipt.prj_id;
                        document.getElementById('edit_type').value = receipt.type;
                        document.getElementById('edit_bank_account').value = receipt.bank_account;

                        // Open modal
                        openEditReceiptModal();
                    } else {
                        alert('{{ __("Receipt not found") }}');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('{{ __("An error occurred") }}');
                });
        }

        function deleteReceipt(id) {
            if (confirm('{{ __("Are you sure you want to delete this receipt?") }}')) {
                fetch(`{{ url('admin/clients/receipts') }}/${id}`, {
                    method: 'DELETE',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            location.reload();
                        }
                    });
            }
        }

        // Add Receipt Form Submission
        document.getElementById('addReceiptForm').addEventListener('submit', function (e) {
            e.preventDefault();

            const formData = new FormData(this);
            const data = Object.fromEntries(formData);

            fetch(`{{ route('clients.receipts.add', $client->cl_id) }}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify(data)
            })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('{{ __("Receipt added successfully") }}');
                        location.reload();
                    } else if (data.error) {
                        alert(data.error);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('{{ __("An error occurred") }}');
                });
        });

        // Edit Receipt Form Submission
        document.getElementById('editReceiptForm').addEventListener('submit', function (e) {
            e.preventDefault();

            const receiptId = document.getElementById('edit_receipt_id').value;
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);

            // Add _method for Laravel PUT request
            data._method = 'PUT';

            fetch(`{{ url('admin/clients/receipts') }}/${receiptId}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify(data)
            })
                .then(response => {
                    if (!response.ok) {
                        return response.text().then(text => {
                            throw new Error(text || 'Server error');
                        });
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        alert('{{ __("Receipt updated successfully") }}');
                        location.reload();
                    } else if (data.error) {
                        alert(data.error);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    // Try to parse error message if it's JSON
                    try {
                        const errObj = JSON.parse(error.message);
                        if (errObj.message) {
                            alert(errObj.message);
                        } else if (errObj.errors) {
                            // Show validation errors
                            let msg = '';
                            for (const [key, value] of Object.entries(errObj.errors)) {
                                msg += value + '\n';
                            }
                            alert(msg);
                        } else {
                            alert('{{ __("An error occurred") }}');
                        }
                    } catch (e) {
                        alert('{{ __("An error occurred") }}: ' + error.message);
                    }
                });
        });

        // Receipt number validation
        document.getElementById('receipt_number').addEventListener('blur', function () {
            const number = this.value;
            if (number) {
                fetch(`{{ route('clients.receipts.check_number') }}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({ number: number })
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.count > 0) {
                            alert('{{ __("Receipt number already exists") }}');
                            this.value = '';
                            this.focus();
                        }
                    });
            }
        });

        // Bank number validation (check system-wide uniqueness)
        document.getElementById('bnk_receipt_number').addEventListener('blur', function () {
            const number = this.value;
            const receiptNumber = document.getElementById('receipt_number').value;

            // Check if bank number equals receipt number
            if (number && receiptNumber && number === receiptNumber) {
                alert('{{ __("Bank receipt number cannot be the same as receipt number") }}');
                this.value = '';
                this.focus();
                return;
            }

            if (number) {
                fetch('/admin/clients/receipts/check-number', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({ number: number, field: 'bnk_number' })
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.count > 0) {
                            alert('{{ __("Bank receipt number already exists in the system") }}');
                            this.value = '';
                            this.focus();
                        }
                    });
            }
        });
    </script>
@endsection