@extends('layouts.admin')

@section('content')
    <div class="max-w-6xl mx-auto space-y-6">
        <!-- Header Card -->
        <div class="bg-white rounded-2xl shadow-xl overflow-hidden border border-emerald-100">
            <div class="bg-gradient-to-r from-emerald-600 to-teal-600 p-6 flex justify-between items-center text-white">
                <div>
                    <h2 class="text-2xl font-bold">{{ __('Account Statement') }}</h2>
                    <p class="text-emerald-50 text-sm mt-1">{{ $client->arabic_name }} ({{ $client->old_member_no }})</p>
                </div>
                <div class="text-end">
                    <span
                        class="text-xs uppercase tracking-widest text-emerald-200 font-bold">{{ __('Total Balance') }}</span>
                    <div class="text-3xl font-black mt-1">{{ number_format($sum, 2) }} <span
                            class="text-lg font-normal">{{ __('EGP') }}</span></div>
                </div>
            </div>

            <div class="p-8">
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    <!-- Data Selection Column -->
                    <div class="lg:col-span-2 space-y-6">
                        <div class="flex justify-between items-center">
                            <h3 class="text-lg font-bold text-gray-800 flex items-center">
                                <i class="entypo-list text-emerald-500 mr-2 text-xl"></i> {{ __('Available Receipts') }}
                            </h3>
                            <div class="text-xs text-gray-500 font-medium bg-gray-100 px-3 py-1 rounded-full">
                                {{ count($receipts) }} {{ __('Receipts Found') }}
                            </div>
                        </div>

                        <div class="border border-gray-100 rounded-xl overflow-hidden shadow-sm bg-gray-50/30">
                            <table class="min-w-full divide-y divide-gray-200" id="receiptsTable">
                                <thead class="bg-white">
                                    <tr class="text-xs font-bold text-gray-500 uppercase">
                                        <th class="px-4 py-3 text-center">
                                            <input type="checkbox" id="selectAll"
                                                class="rounded border-gray-300 text-emerald-600 focus:ring-emerald-500">
                                        </th>
                                        <th class="px-4 py-3 text-start">{{ __('Receipt No.') }}</th>
                                        <th class="px-4 py-3 text-center" style="min-width: 120px;">{{ __('Date') }}</th>
                                        <th class="px-4 py-3 text-end">{{ __('Amount') }}</th>
                                    </tr>
                                </thead>
                                <tbody class="divide-y divide-gray-100 bg-white">
                                    @forelse($receipts as $r)
                                        <tr class="hover:bg-emerald-50/50 transition-colors group row_{{ $r->id }}">
                                            <td class="px-4 py-3 text-center">
                                                <input type="checkbox"
                                                    class="receipt-chk rounded border-gray-300 text-emerald-600 focus:ring-emerald-500"
                                                    data-id="{{ $r->id }}" data-value="{{ $r->value }}">
                                            </td>
                                            <td class="px-4 py-3 text-sm font-semibold text-gray-700">{{ $r->number }}</td>
                                            <td class="px-4 py-3 text-center text-sm text-gray-500" style="min-width: 120px;">
                                                {{ $r->date }}
                                            </td>
                                            <td class="px-4 py-3 text-end text-sm font-black text-emerald-600">
                                                {{ number_format($r->value, 2) }}
                                            </td>
                                        </tr>
                                    @empty
                                        <tr>
                                            <td colspan="4" class="px-4 py-10 text-center text-gray-400 italic">
                                                {{ __('No active receipts found.') }}
                                            </td>
                                        </tr>
                                    @endforelse
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- Action Panel -->
                    <div class="bg-gray-50 rounded-2xl p-6 border border-gray-100 space-y-6 self-start">
                        <h3 class="text-lg font-bold text-gray-800 flex items-center">
                            <i class="entypo-tools text-emerald-500 mr-2 text-xl"></i> {{ __('Operations') }}
                        </h3>

                        <div class="space-y-4">
                            <div>
                                <label
                                    class="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-2">{{ __('Operation Type') }}</label>
                                <select id="operationType"
                                    class="w-full rounded-xl border-gray-200 text-sm focus:ring-emerald-500 focus:border-emerald-500 shadow-sm">
                                    <option value="">{{ __('Select Operation...') }}</option>
                                    <option value="transfer">{{ __('Transfer Funds') }}</option>
                                    <option value="refund">{{ __('Cash Refund') }}</option>
                                </select>
                            </div>

                            <!-- Transfer Section -->
                            <div id="transferSection" class="hidden space-y-4 animate-in fade-in slide-in-from-top-4">
                                <div class="p-4 bg-blue-50/50 rounded-xl border border-blue-100 space-y-3">
                                    <div>
                                        <label
                                            class="block text-xs font-bold text-blue-600 uppercase mb-1">{{ __('Destination Member No.') }}</label>
                                        <input type="text" id="destMemberNo"
                                            class="w-full rounded-lg border-blue-200 text-sm focus:ring-blue-500 focus:border-blue-500"
                                            placeholder="{{ __('Type ID...') }}">
                                        <div id="destClientInfo"
                                            class="mt-2 text-xs font-bold text-blue-700 hidden py-1 px-2 bg-blue-100 rounded">
                                        </div>
                                    </div>
                                    <div>
                                        <label
                                            class="block text-xs font-bold text-blue-600 uppercase mb-1">{{ __('Project') }}</label>
                                        <select id="transferProject"
                                            class="w-full rounded-lg border-blue-200 text-sm focus:ring-blue-500 focus:border-blue-500">
                                            <!-- Projects will be loaded via AJAX like legacy -->
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <!-- Refund Section -->
                            <div id="refundSection" class="hidden space-y-4 animate-in fade-in slide-in-from-top-4">
                                <div class="p-4 bg-orange-50/50 rounded-xl border border-orange-100 space-y-3">
                                    <div>
                                        <label
                                            class="block text-xs font-bold text-orange-600 uppercase mb-1">{{ __('Check Number') }}</label>
                                        <input type="text" id="checkNumber"
                                            class="w-full rounded-lg border-orange-200 text-sm focus:ring-orange-500 focus:border-orange-500">
                                    </div>
                                    <div>
                                        <label
                                            class="block text-xs font-bold text-orange-600 uppercase mb-1">{{ __('Check Date') }}</label>
                                        <input type="date" id="checkDate"
                                            class="w-full rounded-lg border-orange-200 text-sm focus:ring-orange-500 focus:border-orange-500">
                                    </div>
                                    <div>
                                        <label
                                            class="block text-xs font-bold text-orange-600 uppercase mb-1">{{ __('Bank Account') }}</label>
                                        <select id="bankAccount"
                                            class="w-full rounded-lg border-orange-200 text-sm focus:ring-orange-500 focus:border-orange-500">
                                            @foreach($banks as $bank)
                                                <option value="{{ $bank->bank_id }}">{{ $bank->bank_name }} -
                                                    {{ $bank->account_number }}
                                                </option>
                                            @endforeach
                                        </select>
                                    </div>
                                    <div>
                                        <label
                                            class="block text-xs font-bold text-orange-600 uppercase mb-1">{{ __('Associated Project') }}</label>
                                        <select id="refundProject"
                                            class="w-full rounded-lg border-orange-200 text-sm focus:ring-orange-500 focus:border-orange-500">
                                            @foreach($reservations as $res)
                                                <option value="{{ $res->prj_id }}">{{ $res->name }}</option>
                                            @endforeach
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <!-- Summary Column in Action Panel -->
                            <div class="pt-4 border-t border-gray-200">
                                <div
                                    class="flex justify-between items-center mb-1 text-xs text-gray-500 font-bold uppercase tracking-wider">
                                    <span>{{ __('Selected Amount') }}</span>
                                </div>
                                <div class="text-3xl font-black text-gray-900 mb-6" id="selectedAmountDisplay">
                                    0.00 <span class="text-xs font-normal text-gray-500">EGP</span>
                                </div>

                                <button id="processAction"
                                    class="w-full py-4 bg-emerald-600 text-white rounded-2xl font-bold text-lg hover:bg-emerald-700 transition duration-150 shadow-lg shadow-emerald-200 flex items-center justify-center space-x-2 disabled:bg-gray-300 disabled:shadow-none"
                                    disabled>
                                    <i class="entypo-check mr-2"></i> {{ __('Process Operation') }}
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <style>
        @media print {
            .hid {
                display: none !important;
            }
        }
    </style>

    <script>
        document.addEventListener('DOMContentLoaded', function () {
            const operationTypeSelect = document.getElementById('operationType');
            const transferSection = document.getElementById('transferSection');
            const refundSection = document.getElementById('refundSection');
            const processButton = document.getElementById('processAction');
            const selectAll = document.getElementById('selectAll');
            const checkboxes = document.querySelectorAll('.receipt-chk');
            const amountDisplay = document.getElementById('selectedAmountDisplay');
            const destMemberNoInput = document.getElementById('destMemberNo');
            const destClientInfo = document.getElementById('destClientInfo');
            const transferProjectSelect = document.getElementById('transferProject');

            let selectedTotal = 0;

            function calculateTotal() {
                selectedTotal = 0;
                document.querySelectorAll('.receipt-chk:checked').forEach(cb => {
                    selectedTotal += parseFloat(cb.dataset.value);
                });
                updateUI();
            }

            function updateUI() {
                const opType = operationTypeSelect.value;
                transferSection.classList.add('hidden');
                refundSection.classList.add('hidden');

                if (opType === 'transfer') transferSection.classList.remove('hidden');
                else if (opType === 'refund') refundSection.classList.remove('hidden');

                amountDisplay.innerText = selectedTotal.toLocaleString(undefined, { minimumFractionDigits: 2 }) + ' EGP';

                // Enable button only if type selected and amount > 0
                processButton.disabled = !opType || selectedTotal <= 0;
            }

            operationTypeSelect.addEventListener('change', updateUI);

            selectAll.addEventListener('change', function () {
                checkboxes.forEach(cb => {
                    cb.checked = this.checked;
                });
                calculateTotal();
            });

            checkboxes.forEach(cb => {
                cb.addEventListener('change', calculateTotal);
            });

            // Destination member lookup
            destMemberNoInput.addEventListener('blur', function () {
                const val = this.value;
                if (!val) return;

                // Search by cl_id, member_no or old_member_no
                fetch(`{{ route('clients.ajax_search') }}?term=${val}`)
                    .then(res => res.json())
                    .then(data => {
                        // Find match
                        const client = data.find(c => c.old_member_no == val || c.member_no == val || c.cl_id == val);
                        if (client) {
                            destClientInfo.innerText = '{{ __('Name') }}: ' + client.arabic_name;
                            destClientInfo.classList.remove('hidden');

                            // Load projects for this client
                            fetch(`{{ url('admin/clients') }}/${client.cl_id}/reservations-data`)
                                .then(res => res.json())
                                .then(prjData => {
                                    const resv = prjData.reservations || [];
                                    transferProjectSelect.innerHTML = resv.length ? resv.map(p => `<option value="${p.prj_id}">${p.project_name} - ${p.bldg_id}/${p.flat_num}</option>`).join('') : '<option value="">{{ __("No reservations found") }}</option>';
                                });
                        } else {
                            alert('{{ __('Membership number not found') }}');
                            this.value = '';
                            destClientInfo.classList.add('hidden');
                            transferProjectSelect.innerHTML = '';
                        }
                    });
            });

            processButton.addEventListener('click', function () {
                const op = operationTypeSelect.value;
                const confirmMsg = op === 'refund' ? '{{ __('Are you sure you want to process this refund and potentially clear allocations?') }}' : '{{ __('Are you sure you want to process this fund transfer?') }}';

                if (!confirm(confirmMsg)) return;

                const selectedIds = Array.from(document.querySelectorAll('.receipt-chk:checked')).map(cb => cb.dataset.id);

                // Get destination info
                const targetMemNo = destMemberNoInput.value;
                const targetPrjId = transferProjectSelect.value;

                if (op === 'transfer' && (!targetMemNo || !targetPrjId)) {
                    alert('{{ __("Please select a target client and project.") }}');
                    return;
                }

                // Show processing
                processButton.disabled = true;
                processButton.innerText = '{{ __("Processing...") }}';

                fetch('{{ route('clients.transaction.save') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({
                        amount: selectedTotal,
                        mem_no: targetMemNo,
                        from_id: '{{ $client->cl_id }}',
                        base_amount: '{{ $sum }}', // Original balance
                        prj_id: targetPrjId || '{{ $receipts->first()->prj_id ?? 0 }}',
                        rec_ids: selectedIds.join('|') + '|',
                        type: op === 'transfer' ? 'to_cl' : 'to_chk',
                        // Refund fields if needed
                        chk_num: document.getElementById('checkNumber').value,
                        chk_date: document.getElementById('checkDate').value,
                        chk_bnk: document.getElementById('bankAccount').value
                    })
                })
                    .then(res => res.json())
                    .then(data => {
                        if (data.success) {
                            alert(data.message || '{{ __("Operation completed successfully") }}');
                            location.reload();
                        } else {
                            throw new Error(data.error || data.message || 'Error occurred');
                        }
                    })
                    .catch(err => {
                        alert('Error: ' + err.message);
                        processButton.disabled = false;
                        processButton.innerText = '{{ __("Process Operation") }}';
                    });
            });
        });
    </script>
@endsection