<?php

function parseSqlFile($filePath)
{
    if (!file_exists($filePath)) {
        die("File not found: $filePath\n");
    }

    $content = file_get_contents($filePath);

    // Convert from UTF-16LE (common in Windows PowerShell output) to UTF-8 if needed
    if (strpos($content, "\xFF\xFE") === 0) {
        if (function_exists('mb_convert_encoding')) {
            $content = mb_convert_encoding($content, 'UTF-8', 'UTF-16LE');
        } else {
            // Fallback: strip null bytes (works for ASCII/English)
            $content = str_replace("\0", "", $content);
        }
    }
    // Also try simple UTF-8 detection or just ensure it's valid
    // Note: mysqldump might output without BOM but assuming UTF-8 usually works if not UTF-16

    $tables = [];

    // Improved regex to capture table name and body including IF NOT EXISTS
    if (preg_match_all('/CREATE\s+TABLE\s+(?:IF\s+NOT\s+EXISTS\s+)?[`"]?(\w+)[`"]?\s*\((.*?)\)\s*ENGINE/si', $content, $matches, PREG_SET_ORDER)) {
        foreach ($matches as $match) {
            $tableName = $match[1];
            $body = $match[2];
            $columns = [];
            $primaryKeys = [];

            // Split by comma, but respect parentheses (for DECIMAL(10,2) or ENUM(...))
            $lines = preg_split("/,(?![^()]*\))/", $body);

            foreach ($lines as $line) {
                $line = trim($line);
                if (empty($line))
                    continue;

                // Check for Primary Key definition at end
                if (preg_match('/^PRIMARY\s+KEY\s*\((.*?)\)/i', $line, $pkMatch)) {
                    // Extract columns
                    preg_match_all('/[`"]?(\w+)[`"]?/', $pkMatch[1], $cols);
                    $primaryKeys = $cols[1];
                    continue;
                }

                // Skip other keys for now to focus on columns
                if (preg_match('/^(KEY|UNIQUE|CONSTRAINT|FULLTEXT|FOREIGN)/i', $line)) {
                    continue;
                }

                // Parse Column Definition
                // `col_name` type options...
                if (preg_match('/^[`"]?(\w+)[`"]?\s+(.*)$/', $line, $colMatch)) {
                    $colName = $colMatch[1];
                    $definition = $colMatch[2];
                    $columns[$colName] = $definition;
                }
            }
            $tables[$tableName] = [
                'columns' => $columns,
                'full_statement' => "CREATE TABLE `$tableName` (\n$body\n) ENGINE=InnoDB DEFAULT CHARSET=utf8;"
                // Note: simplified engine/charset for new tables, might want to capture original if possible.
                // Actually, let's grab the original full CREATE block from the file for new tables.
            ];
        }
    }

    // Second pass to capture full CREATE statements correctly for new tables
    // The previous regex stopped at ENGINE, so let's use a simpler approach for full bodies if needed
    // But for "Add Column", the above parsed columns are what matters.
    // For "Create Table", we can just extract the block again or use the generated one. 
    // Let's refine the "Full Statement" capture.

    return $tables;
}

function getFullCreateStatement($filePath, $tableName)
{
    $content = file_get_contents($filePath);
    // Capture from CREATE TABLE table_name ... until ;
    // This is rough but should work for mysqldump outputs
    if (preg_match("/CREATE\s+TABLE\s+(?:IF\s+NOT\s+EXISTS\s+)?[`\"]?$tableName+[`\"]?.*?;/si", $content, $match)) {
        return $match[0];
    }
    return "";
}

$oldTables = parseSqlFile('old.sql');
$newTables = parseSqlFile('current_schema.sql');

$sql = "-- Database Update Script\n";
$sql .= "-- Generated execution time: " . date('Y-m-d H:i:s') . "\n\n";

// 1. New Tables
foreach ($newTables as $name => $struct) {
    if (!isset($oldTables[$name])) {
        // Use the parsed "full content" which we constructed during parsing.
        // Wait, parseSqlFile doesn't expose the original body... let's check parseSqlFile.
        // Actually, parseSqlFile DOES construct 'full_statement' in line 64!
        // So we can just use that directly.
        $stmt = $struct['full_statement'];

        if ($stmt) {
            $sql .= "-- New Table: $name\n";
            $sql .= $stmt . "\n\n";
        } else {
            $sql .= "-- ERR: Could not extract CREATE stmt for $name\n";
        }
    }
}

// 2. Modified Tables
foreach ($newTables as $name => $newStruct) {
    if (isset($oldTables[$name])) {
        $oldCols = $oldTables[$name]['columns'];
        $newCols = $newStruct['columns'];

        $addedCols = [];
        $modifiedCols = [];

        foreach ($newCols as $colName => $def) {
            if (!isset($oldCols[$colName])) {
                $addedCols[] = "ADD COLUMN `$colName` $def";
            } else {
                // Compare definitions roughly
                // Normalize spaces
                $oldDef = preg_replace('/\s+/', ' ', strtolower(trim($oldCols[$colName])));
                $newDef = preg_replace('/\s+/', ' ', strtolower(trim($def)));

                // Ignore harmless diffs like 'int(11)' vs 'int' if desired, but strict for now
                if ($oldDef !== $newDef) {
                    // $modifiedCols[] = "MODIFY COLUMN `$colName` $def";
                    // For now, let's just log added columns to stay safe. 
                    // Modifying columns automatically is risky without data migration logic.
                }
            }
        }

        if (!empty($addedCols)) {
            $sql .= "-- Update Table: $name\n";
            $sql .= "ALTER TABLE `$name`\n" . implode(",\n", $addedCols) . ";\n\n";
        }
    }
}

echo "Columns of inst_table:\n";
if (isset($newTables['inst_table'])) {
    print_r($newTables['inst_table']['columns']);
} else {
    echo "inst_table not found in parsed tables.\n";
}

file_put_contents('production_update.sql', $sql);
echo "Analysis Complete. Script generated at production_update.sql\n";

?>