<?php

namespace App\Services;

class ArabicNumberService
{
    /**
     * Converts a number to Arabic words.
     * Simplified version of I18N_Arabic_Numbers::money2str specifically for EGP.
     * 
     * @param float $number
     * @return string
     */
    public function moneyToArabicWords($number)
    {
        // For now, let's implement a very basic version or use a simpler logic
        // since the full XML-based logic is heavy.
        // Given the requirement is for payment orders in EGP.

        $number = round($number, 2);
        $parts = explode('.', (string) $number);
        $pounds = (int) $parts[0];
        $piastres = isset($parts[1]) ? (int) str_pad($parts[1], 2, '0', STR_PAD_RIGHT) : 0;

        $words = $this->numberToWords($pounds) . ' جنيهاً مصرياً';

        if ($piastres > 0) {
            $words .= ' و ' . $this->numberToWords($piastres) . ' قرشاً';
        }

        return $words . ' فقط لا غير';
    }

    private function numberToWords($number)
    {
        if ($number == 0)
            return 'صفر';

        $hyphen = ' ';
        $conjunction = ' و ';
        $separator = ' ';
        $negative = 'سالب ';
        $decimal = ' فاصلة ';
        $dictionary = array(
            0 => 'صفر',
            1 => 'واحد',
            2 => 'اثنان',
            3 => 'ثلاثة',
            4 => 'أربعة',
            5 => 'خمسة',
            6 => 'ستة',
            7 => 'سبعة',
            8 => 'ثمانية',
            9 => 'تسعة',
            10 => 'عشرة',
            11 => 'أحد عشر',
            12 => 'اثنا عشر',
            13 => 'ثلاثة عشر',
            14 => 'أربعة عشر',
            15 => 'خمسة عشر',
            16 => 'ستة عشر',
            17 => 'سبعة عشر',
            18 => 'ثمانية عشر',
            19 => 'تسعة عشر',
            20 => 'عشرون',
            30 => 'ثلاثون',
            40 => 'أربعون',
            50 => 'خمسون',
            60 => 'ستون',
            70 => 'سبعون',
            80 => 'ثمانون',
            90 => 'تسعون',
            100 => 'مائة',
            200 => 'مائتان',
            300 => 'ثلاثمائة',
            400 => 'أربعمائة',
            500 => 'خمسمائة',
            600 => 'ستمائة',
            700 => 'سبعمائة',
            800 => 'ثمانمائة',
            900 => 'تسعمائة',
            1000 => 'ألف',
            2000 => 'ألفان',
            1000000 => 'مليون',
            1000000000 => 'مليار'
        );

        if (!is_numeric($number))
            return false;

        if ($number < 0)
            return $negative . $this->numberToWords(abs($number));

        $string = null;

        switch (true) {
            case $number < 21:
                $string = $dictionary[$number];
                break;
            case $number < 100:
                $tens = ((int) ($number / 10)) * 10;
                $units = $number % 10;
                $string = $dictionary[$tens];
                if ($units) {
                    $string = $dictionary[$units] . $conjunction . $string;
                }
                break;
            case $number < 1000:
                $hundreds = ((int) ($number / 100)) * 100;
                $remainder = $number % 100;
                $string = $dictionary[$hundreds];
                if ($remainder) {
                    $string .= $conjunction . $this->numberToWords($remainder);
                }
                break;
            case $number < 1000000:
                $thousands = (int) ($number / 1000);
                $remainder = $number % 1000;

                if ($thousands == 1) {
                    $string = $dictionary[1000];
                } elseif ($thousands == 2) {
                    $string = $dictionary[2000];
                } elseif ($thousands >= 3 && $thousands <= 10) {
                    $string = $this->numberToWords($thousands) . ' آلاف';
                } else {
                    $string = $this->numberToWords($thousands) . ' ألف';
                }

                if ($remainder) {
                    $string .= $conjunction . $this->numberToWords($remainder);
                }
                break;
            default:
                // Handle millions and billions if needed, but for most payments this is enough
                $string = (string) $number;
                break;
        }

        return $string;
    }
}
