<?php

namespace App\Http\Controllers;

use App\Models\Project;
use App\Models\Client;
use App\Services\ClientAccountService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportsController extends Controller
{
    protected $accountService;

    public function __construct(ClientAccountService $accountService)
    {
        $this->accountService = $accountService;
    }

    public function inquiryByBuilding()
    {
        $projects = Project::where('flag', 'Y')->get();
        return view('admin.reports.inquiry_by_building', compact('projects'));
    }

    public function parkingAndGarden()
    {
        $projects = Project::where('flag', 'Y')->get();
        return view('admin.reports.parking_and_garden', compact('projects'));
    }

    public function getBuildings(Request $request)
    {
        $prj_id = $request->prj_id;
        $buildings = DB::table('default_reservations')
            ->where('prj_id', $prj_id)
            ->whereNotNull('bldg_id')
            ->where('bldg_id', '!=', '0')
            ->distinct()
            ->select('bldg_id')
            ->get();

        $options = '<option value="0">' . __('All Buildings') . '</option>';

        // Handle special logic for project 2 from legacy
        if ($prj_id == '2') {
            foreach ($buildings as $row) {
                if ($row->bldg_id <= 16) {
                    $options .= '<option value="A-' . $row->bldg_id . '">A-' . $row->bldg_id . '</option>';
                    $options .= '<option value="B-' . $row->bldg_id . '">B-' . $row->bldg_id . '</option>';
                } else {
                    $options .= '<option value="' . $row->bldg_id . '">' . $row->bldg_id . '</option>';
                }
            }
        } else {
            foreach ($buildings as $row) {
                $options .= '<option value="' . $row->bldg_id . '">' . $row->bldg_id . '</option>';
            }
        }

        return response($options);
    }

    public function getFlats(Request $request)
    {
        $prj_id = $request->prj_id;
        $flats = DB::table('flats')
            ->where('prj_id', $prj_id)
            ->distinct()
            ->select('flt_id', 'txt')
            ->orderBy('txt')
            ->get();

        $options = '<option value="0">' . __('All Types') . '</option>';
        foreach ($flats as $row) {
            $options .= '<option value="' . $row->flt_id . '">' . $row->txt . '</option>';
        }

        return response($options);
    }

    public function getAreas(Request $request)
    {
        $prj_id = $request->prj_id;
        $areas = DB::table('flats')
            ->where('prj_id', $prj_id)
            ->whereNotNull('area')
            ->where('area', '!=', '')
            ->distinct()
            ->select('area')
            ->get();

        $options = '<option value="0">' . __('All Areas') . '</option>';
        foreach ($areas as $row) {
            $area_label = $row->area;
            if ($area_label == 'sepvila' || $area_label == 'sep_vila')
                $area_label = 'فيلا منفصله';
            if ($area_label == 'twinvila')
                $area_label = 'توين فيلا';
            if ($area_label == 'grand_265')
                $area_label = 'جراند 265';

            $options .= '<option value="' . $row->area . '">' . $area_label . '</option>';
        }

        return response($options);
    }

    public function inquiryByBuildingData(Request $request)
    {
        $prj_id = $request->prj_id;
        $bldg_id = $request->bldg_id;
        $flt_id = $request->flt_id;
        $area_filter = $request->area;

        $query = DB::table('default_reservations as rs')
            ->leftJoin('default_clients as cl', 'cl.member_no', '=', 'rs.cl_id')
            ->leftJoin('flats as fl', 'fl.flt_id', '=', 'rs.flt_id')
            ->select(
                'cl.cl_id',
                'rs.flt_id',
                'rs.prj_id',
                'rs.floor',
                'rs.flat_num',
                'fl.area',
                'cl.arabic_name',
                'cl.old_member_no',
                'cl.member_no',
                'cl.file_no',
                'rs.bldg_id',
                'cl.address',
                'cl.mobile',
                'rs.sess',
                'rs.subsess'
            )
            ->where('rs.prj_id', $prj_id)
            ->where('cl.flag', 'Y')
            ->whereNotNull('cl.member_no')
            ->where('cl.member_no', '!=', '0')
            ->where('cl.member_no', '!=', '');

        if ($bldg_id && $bldg_id != '0') {
            // Handle project 2 prefix if needed
            $actual_bldg = $bldg_id;
            if ($prj_id == '2' && str_contains($bldg_id, '-')) {
                $parts = explode('-', $bldg_id);
                $actual_bldg = $parts[1];
            }
            $query->where('rs.bldg_id', $actual_bldg);
        }

        if ($flt_id && $flt_id != '0') {
            $query->where('rs.flt_id', $flt_id);
        }

        if ($area_filter && $area_filter != '0') {
            $query->where('fl.area', $area_filter);
        }

        $results = $query->get();

        $data = [];
        foreach ($results as $row) {
            $money = $this->accountService->calculateAccountReport($row->cl_id, $row->prj_id);

            $area = $row->area;
            if ($area == 'sepvila' || $area == 'sep_vila')
                $area = 'فيلا منفصله';
            if ($area == 'twinvila')
                $area = 'توين فيلا';
            if ($area == 'grand_265')
                $area = 'جراند 265';

            $paid = $money['total'] - ($money['park_total'] ?? 0);
            if ($paid < 0)
                $paid = abs($paid);

            $data[] = [
                'bldg' => $row->bldg_id,
                'floor' => $row->floor,
                'flat_num' => $row->flat_num,
                'area' => $area,
                'name' => $row->arabic_name,
                'old_member_no' => $row->old_member_no,
                'file_no' => $row->file_no,
                'mobile' => $row->mobile,
                'address' => $row->address,
                'required' => number_format($money['biggest_total'] ?? 0, 2),
                'paid' => number_format($paid, 2),
                'remaining' => number_format($money['remain'] ?? 0, 2),
                'phase' => $row->sess == '111' ? 'بحريه' : $row->sess,
                'subsess' => $row->subsess > 0 ? 'نعم' : 'لا'
            ];
        }

        return response()->json(['data' => $data]);
    }

    public function parkingAndGardenData(Request $request)
    {
        $prj_id = $request->prj_id;
        // Use Ym format (leading zero for month) for reliable numeric/string comparison
        $currentYrMon = date('Ym');

        // 1. Bulk fetch parking installments
        $parkingInst = DB::table('default_park_inst')
            ->where('prj_id', $prj_id)
            ->get()
            ->groupBy(['bldg_no', 'park_no']);

        // 2. Bulk fetch garden installments
        $gardenInst = DB::table('garden_inst_table')
            ->where('prj_id', $prj_id)
            ->get()
            ->groupBy(['bldg_no', 'flat_num']);

        // 3. Fetch reservations with garden or park
        $reservations = DB::table('default_reservations as rs')
            ->leftJoin('default_clients as cl', 'cl.member_no', '=', 'rs.cl_id')
            ->leftJoin('default_park_resv as p', function ($join) use ($prj_id) {
                $join->on('p.cl_id', '=', 'rs.cl_id')
                    ->where('p.prj_id', '=', $prj_id);
            })
            ->select(
                'rs.*',
                'p.park_no',
                'p.bldg as park_bldg',
                'cl.cl_id as clid',
                'cl.old_member_no',
                'cl.arabic_name'
            )
            ->where('rs.prj_id', $prj_id)
            ->where(function ($query) {
                $query->where('rs.garden', 'Y')
                    ->orWhere('rs.park', 'Y');
            })
            ->where('cl.flag', 'Y')
            ->get();

        $data = [];
        foreach ($reservations as $res) {
            // Paid amounts should use cl.cl_id
            $park_paid = DB::table('default_receipts')
                ->where('prj_id', $prj_id)
                ->where('cl_id', $res->clid)
                ->where('type', '3')
                ->where('flag', 'Y')
                ->where('chk', 'N')
                ->sum('value');

            $garden_paid = DB::table('default_receipts')
                ->where('prj_id', $prj_id)
                ->where('cl_id', $res->clid)
                ->where('type', '8')
                ->where('flag', 'Y')
                ->where('chk', 'N')
                ->sum('value');

            $park_required = 0;
            $park_total = 0;
            // For parking, installments are linked to the parking building/spot, not necessarily the unit building
            $p_bldg = $res->park_bldg ?? $res->bldg_id;
            if ($res->park == 'Y' && $res->park_no && isset($parkingInst[$p_bldg][$res->park_no])) {
                $insts = $parkingInst[$p_bldg][$res->park_no];
                $park_total = $insts->sum('inst_amount');
                $park_required = $insts->filter(function ($item) use ($currentYrMon) {
                    // Convert both to integers for comparison if they are Ym format
                    return (int) $item->yr_mon <= (int) $currentYrMon;
                })->sum('inst_amount');
            }

            $garden_required = 0;
            $garden_total = 0;
            if ($res->garden == 'Y' && isset($gardenInst[$res->bldg_id][$res->flat_num])) {
                $insts = $gardenInst[$res->bldg_id][$res->flat_num];
                $garden_total = $insts->sum('gard_inst_amount');
                $garden_required = $insts->filter(function ($item) use ($currentYrMon) {
                    return (int) $item->yr_mon <= (int) $currentYrMon;
                })->sum('gard_inst_amount');
            }

            $park_remain = max(0, $park_required - $park_paid);
            $garden_remain = max(0, $garden_required - $garden_paid);

            $data[] = [
                'old_member_no' => $res->old_member_no,
                'arabic_name' => $res->arabic_name,
                'bldg_id' => $res->bldg_id,
                'park_no' => $res->park_no ?? 0,
                'park_required' => number_format($park_required, 2),
                'park_total' => number_format($park_total, 2),
                'park_paied' => number_format($park_paid, 2),
                'park_remain' => number_format($park_remain, 2),
                'garden_required' => number_format($garden_required, 2),
                'garden_total' => number_format($garden_total, 2),
                'garden_paied' => number_format($garden_paid, 2),
                'garden_remain' => number_format($garden_remain, 2),
            ];
        }

        return response()->json(['data' => $data]);
    }

    /**
     * Project Payment Details Report - View
     */
    public function projectPaymentDetails()
    {
        $projects = Project::where('flag', 'Y')->get();

        // Get all clients for dropdown
        $clients = Client::where('flag', 'Y')
            ->whereNotNull('member_no')
            ->where('member_no', '!=', '0')
            ->where('member_no', '!=', '')
            ->orderBy('arabic_name')
            ->get();

        return view('admin.reports.project_payment_details', compact('projects', 'clients'));
    }

    /**
     * Project Payment Details Report - Data
     */
    public function projectPaymentDetailsData(Request $request)
    {
        $prj_id = $request->prj_id;
        $date_from = $request->date_from;
        $date_to = $request->date_to;
        $cl_id = $request->cl_id;

        // Build the query
        $query = DB::table('default_receipts as r')
            ->leftJoin('default_clients as cl', 'cl.cl_id', '=', 'r.cl_id')
            ->leftJoin('default_reservations as res', function ($join) use ($prj_id) {
                $join->on('res.cl_id', '=', 'cl.member_no')
                    ->where('res.prj_id', '=', $prj_id);
            })
            ->leftJoin('flats as f', function ($join) use ($prj_id) {
                $join->on('f.flt_id', '=', 'res.flt_id')
                    ->on('f.prj_id', '=', 'res.prj_id');
            })
            ->leftJoin('projects as pr', 'pr.prj_id', '=', 'r.prj_id')
            ->leftJoin('banks as bk', 'bk.bank_id', '=', 'r.bank_account')
            ->leftJoin('installment_type as t', 't.id', '=', 'r.type')
            ->select(
                'cl.file_no',
                'cl.arabic_name',
                'cl.old_member_no',
                'cl.cl_id',
                'cl.member_no',
                'f.txt as flat_name',
                'pr.name as project_name',
                'r.value',
                't.type as receipt_type',
                'r.date',
                'r.number',
                'r.bnk_number',
                'bk.bank_name',
                'bk.account_number'
            )
            ->where('r.prj_id', $prj_id)
            ->where('r.chk', 'N')
            ->where('r.flag', 'Y')
            ->where('r.type', '!=', '111')
            ->where('cl.flag', 'Y')
            ->whereNotNull('cl.member_no')
            ->where('cl.member_no', '!=', '0')
            ->where('cl.member_no', '!=', '');

        // Date filter
        if ($date_from && $date_to) {
            $query->whereBetween('r.rel_date', [
                date('Y-m-d', strtotime($date_from)),
                date('Y-m-d', strtotime($date_to))
            ]);
        }

        // Client filter
        if ($cl_id && $cl_id != 'all') {
            $query->where('cl.cl_id', $cl_id);
        }

        $query->orderBy('cl.file_no');

        $results = $query->get();

        $total = 0;
        $data = [];

        foreach ($results as $row) {
            // Only add rows that have a flat name
            if ($row->flat_name && $row->flat_name != null) {
                $data[] = [
                    'file_no' => $row->file_no ?? '',
                    'arabic_name' => $row->arabic_name ?? '',
                    'old_member_no' => $row->old_member_no ?? '',
                    'flat_name' => $row->flat_name ?? '',
                    'project_name' => $row->project_name ?? '',
                    'value' => number_format($row->value ?? 0),
                    'receipt_type' => $row->receipt_type ?? '',
                    'date' => $row->date ?? '',
                    'number' => $row->number ?? '',
                    'bnk_number' => $row->bnk_number ?? '',
                    'bank_name' => $row->bank_name ?? '',
                    'account_number' => $row->account_number ?? ''
                ];
                $total += $row->value ?? 0;
            }
        }

        return response()->json([
            'data' => $data,
            'total' => number_format($total)
        ]);
    }

    /**
     * Penalties Report - View
     */
    public function penaltiesReport()
    {
        $projects = Project::where('flag', 'Y')->get();
        return view('admin.reports.penalties', compact('projects'));
    }

    /**
     * Penalties Report - Data
     */
    public function penaltiesReportData(Request $request)
    {
        $prj_id = $request->prj_id;

        $query = DB::table('default_receipts as r')
            ->leftJoin('default_clients as c', 'c.cl_id', '=', 'r.cl_id')
            ->leftJoin('projects as p', 'p.prj_id', '=', 'r.prj_id')
            ->leftJoin('banks as bk', 'bk.bank_id', '=', 'r.bank_account')
            ->select(
                'r.date',
                'c.old_member_no',
                'c.arabic_name',
                'c.file_no',
                'r.value',
                'r.number',
                'bk.account_number',
                'p.name as project_name'
            )
            ->where('r.flag', 'Y')
            ->where('r.type', '100')
            ->where('r.prj_id', $prj_id)
            ->orderByRaw("STR_TO_DATE(REPLACE(r.date,'-',','),'%d,%m,%Y') ASC");

        $results = $query->get();

        $total = 0;
        $data = [];

        foreach ($results as $row) {
            $data[] = [
                'date' => $row->date ?? '',
                'old_member_no' => $row->old_member_no ?? '',
                'arabic_name' => $row->arabic_name ?? '',
                'file_no' => $row->file_no ?? '',
                'value' => number_format($row->value ?? 0),
                'number' => $row->number ?? '',
                'account_number' => $row->account_number ?? '',
                'project_name' => $row->project_name ?? ''
            ];
            $total += $row->value ?? 0;
        }

        return response()->json([
            'data' => $data,
            'total' => number_format($total)
        ]);
    }

    /**
     * Receipt Review Report - Employee Errors
     */
    public function receiptReviewReport()
    {
        $query = DB::table('default_receipts as r')
            ->leftJoin('default_employee as e', 'e.emp_id', '=', 'r.reg_by')
            ->select(DB::raw('count(r.id) as cnt'), 'r.reg_by', 'e.arabic_name', 'e.emp_id')
            ->where('r.flag', 'N')
            ->where('r.cnf', 'X')
            ->where('e.job', '!=', 'مدير الموقع')
            ->groupBy('r.reg_by', 'e.arabic_name', 'e.emp_id')
            ->get();

        return view('admin.reports.receipt_review_report', ['employees' => $query]);
    }

    /**
     * Receipt Review Details - AJAX for modal
     */
    public function receiptReviewDetails(Request $request)
    {
        $emp_id = $request->emp_id;

        $receipts = DB::table('default_receipts as r')
            ->leftJoin('default_clients as c', 'c.cl_id', '=', 'r.cl_id')
            ->select('r.*', 'c.old_member_no')
            ->where('r.flag', 'N')
            ->where('r.cnf', 'X')
            ->where('r.reg_by', $emp_id)
            ->get();

        $html = '<table class="table table-bordered" style="width:100%">';
        $html .= '<thead style="background-color:#f97316; color:white;">';
        $html .= '<tr>';
        $html .= '<th style="padding:12px; text-align:center; font-weight:bold;">رقم العضويه</th>';
        $html .= '<th style="padding:12px; text-align:center; font-weight:bold;">رقم الايصال</th>';
        $html .= '<th style="padding:12px; text-align:center; font-weight:bold;">القيمه</th>';
        $html .= '<th style="padding:12px; text-align:center; font-weight:bold;">تاريخ الايصال</th>';
        $html .= '</tr>';
        $html .= '</thead>';
        $html .= '<tbody>';

        foreach ($receipts as $receipt) {
            $html .= '<tr>';
            $html .= '<td style="padding:10px; text-align:center; font-weight:600; color:#f97316;">' . ($receipt->old_member_no ?? '') . '</td>';
            $html .= '<td style="padding:10px; text-align:center;">' . ($receipt->number ?? '') . '</td>';
            $html .= '<td style="padding:10px; text-align:center; font-weight:bold; color:#059669;">' . number_format($receipt->value ?? 0) . '</td>';
            $html .= '<td style="padding:10px; text-align:center;">' . ($receipt->date ?? '') . '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody></table>';

        return response($html);
    }

    /**
     * Pending Receipts View
     */
    public function pendingReceipts()
    {
        $receipts = DB::table('default_clients as v')
            ->leftJoin('default_receipts as b', 'b.cl_id', '=', 'v.cl_id')
            ->leftJoin('banks as bnk', 'bnk.bank_id', '=', 'b.bank_account')
            ->leftJoin('default_admins as ad', 'ad.emp_id', '=', 'b.upd_by')
            ->leftJoin('installment_type as t', 't.id', '=', 'b.type')
            ->select(
                'b.id as rid',
                'v.*',
                'b.date',
                'b.value',
                'b.number',
                'b.type',
                'bnk.bank_name',
                't.type as type_name',
                'ad.name as adname'
            )
            ->where('b.flag', 'Y')
            ->where('b.cnf', 'N')
            ->where('v.flag', 'Y')
            ->orderBy('b.created_at', 'DESC')
            ->limit(250)
            ->get();

        return view('admin.reports.pending_receipts', compact('receipts'));
    }

    /**
     * Confirm or Cancel Receipt
     */
    public function confirmReceipt(Request $request)
    {
        $id = $request->id;
        $status = $request->status; // 'Y' for confirm, 'X' for cancel

        if ($status == 'X') {
            // Cancel: set cnf='X' AND flag='N'
            DB::table('default_receipts')
                ->where('id', $id)
                ->update([
                    'cnf' => 'X',
                    'flag' => 'N'
                ]);
        } else {
            // Confirm: set cnf='Y'
            DB::table('default_receipts')
                ->where('id', $id)
                ->update(['cnf' => 'Y']);
        }

        return response()->json(['success' => true]);
    }

    /**
     * Deleted Items View
     */
    public function deletedItems()
    {
        // Deleted Clients
        $deletedClients = DB::table('default_clients as v')
            ->leftJoin('default_admins as ad', 'ad.emp_id', '=', 'v.upd_by')
            ->select('v.*', 'ad.name as adname')
            ->where('v.flag', 'N')
            ->get();

        // Deleted Receipts
        $deletedReceipts = DB::table('default_receipts as r')
            ->leftJoin('default_clients as c', 'c.cl_id', '=', 'r.cl_id')
            ->leftJoin('default_admins as ad', 'ad.emp_id', '=', 'r.upd_by')
            ->leftJoin('banks as bnk', 'bnk.bank_id', '=', 'r.bank_account')
            ->leftJoin('installment_type as t', 't.id', '=', 'r.type')
            ->select(
                'r.id as rid',
                'r.*',
                'c.old_member_no',
                'c.file_no',
                'c.arabic_name',
                'ad.name as adname',
                'bnk.bank_name',
                't.type as type_name'
            )
            ->where('r.flag', 'N')
            ->get();

        return view('admin.reports.deleted_items', compact('deletedClients', 'deletedReceipts'));
    }

    /**
     * Restore Client
     */
    public function restoreClient(Request $request)
    {
        $cl_id = $request->cl_id;

        DB::table('default_clients')
            ->where('cl_id', $cl_id)
            ->update(['flag' => 'Y']);

        return response()->json(['success' => true]);
    }

    /**
     * Restore Receipt
     */
    public function restoreReceipt(Request $request)
    {
        $id = $request->id;

        DB::table('default_receipts')
            ->where('id', $id)
            ->update(['flag' => 'Y']);

        return response()->json(['success' => true]);
    }
}
