<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules\Password;

class ProfileController extends Controller
{
    /**
     * Display the profile page
     */
    public function index()
    {
        $user = Auth::guard('admin')->user();
        return view('admin.profile', compact('user'));
    }

    /**
     * Update user information (name, email)
     */
    public function updateInfo(Request $request)
    {
        $user = Auth::guard('admin')->user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:default_admins,email,' . $user->emp_id . ',emp_id',
        ]);

        $user->name = $validated['name'];
        $user->email = $validated['email'];
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'Profile information updated successfully!'
        ]);
    }

    /**
     * Update user password
     */
    public function updatePassword(Request $request)
    {
        $user = Auth::guard('admin')->user();

        $validated = $request->validate([
            'current_password' => 'required',
            'new_password' => ['required', 'confirmed', Password::min(8)],
        ]);

        // Verify current password
        if (!Hash::check($validated['current_password'], $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect!'
            ], 422);
        }

        // Update password
        $user->password = Hash::make($validated['new_password']);
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'Password updated successfully!'
        ]);
    }

    /**
     * Update user profile image
     */
    public function updateImage(Request $request)
    {
        $user = Auth::guard('admin')->user();

        $validated = $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048', // 2MB max
        ]);

        // Delete old image if exists
        if ($user->image && Storage::disk('public')->exists($user->image)) {
            Storage::disk('public')->delete($user->image);
        }

        // Store new image
        $imagePath = $request->file('image')->store('profile-images', 'public');

        // Update user record
        $user->image = $imagePath;
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'Profile image updated successfully!',
            'image_url' => Storage::url($imagePath)
        ]);
    }

    /**
     * Delete user profile image
     */
    public function deleteImage(Request $request)
    {
        $user = Auth::guard('admin')->user();

        // Delete image if exists
        if ($user->image && Storage::disk('public')->exists($user->image)) {
            Storage::disk('public')->delete($user->image);
        }

        // Clear image from database
        $user->image = null;
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'Profile image deleted successfully!'
        ]);
    }
}
