<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;


use App\Models\HrAudit;
use App\Models\Client;
use App\Models\PayOrder;
use App\Models\AuditLog;
use App\Models\Project;
use App\Models\Bank;
use App\Services\ClientAccountService;
use App\Services\ArabicNumberService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class PrintOrderController extends Controller
{
    protected $accountService;
    protected $arabicNumberService;

    public function __construct(ClientAccountService $accountService, ArabicNumberService $arabicNumberService)
    {
        $this->accountService = $accountService;
        $this->arabicNumberService = $arabicNumberService;
    }

    /**
     * Get client's units (reservations) for the selection modal
     */
    public function getClientUnits($id)
    {
        $client = Client::where('cl_id', $id)->firstOrFail();

        $reservations = DB::table('default_reservations as r')
            ->leftJoin('flats as f', 'f.flt_id', '=', 'r.flt_id')
            ->leftJoin('projects as p', 'p.prj_id', '=', 'r.prj_id')
            ->where('r.cl_id', $client->member_no)
            ->select('r.prj_id', 'r.flt_id', 'r.flat_num', 'p.name as project_name', 'f.area')
            ->get();

        return response()->json([
            'client' => $client,
            'reservations' => $reservations
        ]);
    }

    /**
     * Get payment details (installments) for a specific unit
     */
    public function getPaymentDetails(Request $request)
    {
        $cl_id = $request->cl_id;
        $prj_id = $request->prj_id;
        $flt_id = $request->flt_id;

        $client = Client::where('cl_id', $cl_id)->firstOrFail();

        // Calculate due amount using Service
        $accountData = $this->accountService->calculateAccountReport($cl_id, $prj_id);

        // Get banks
        $banks = Bank::where('status', 'Y')->get();

        // Project specific bank (legacy logic)
        $prj_bank_map = [
            "1" => "الأهلي المصري 015|1213070236295600015",
            "2" => "مصر|1590199000000124",
            "3" => "الأهلي المصري 018|1213070236295601018"
        ];
        $default_bank = $prj_bank_map[$prj_id] ?? '';

        return response()->json([
            'remain' => $accountData['remain'] ?? 0,
            'banks' => $banks,
            'default_bank' => $default_bank,
            'installment_types' => [
                'قسط شهر فبراير',
                'قسط شهر مايو',
                'قسط شهر اغسطس',
                'قسط شهر نوفمبر',
                'قسط شهر اغسطس و سنوى',
                'غرامه',
                'استكمال',
                'اشتراك',
                'تنازل',
                'تسويه',
                'جراج',
                'روف',
                'حديقه',
                'متاخرات',
                'تخصيص',
                'عدادات',
                'فوائد بنكيه',
                'مقدم',
                'دفعه أستلام',
                'تأمين',
                'كهرباء',
                'غاز',
                'مياه'
            ]
        ]);
    }

    /**
     * Store the print order and log the audit
     */
    public function store(Request $request)
    {
        $request->validate([
            'cl_id' => 'required',
            'prj_id' => 'required',
            'amount' => 'required|numeric',
        ]);

        $client = Client::where('cl_id', $request->cl_id)->firstOrFail();
        $admin = Auth::guard('admin')->user();

        // Save to default_pay_order
        $order = PayOrder::create([
            'member_no' => $client->member_no,
            'amount' => $request->amount,
            'bank' => $request->bank,
            'type' => $request->inst_type,
            'issue_date' => now()->toDateTimeString(),
            'prj_id' => $request->prj_id,
            'pay_until' => $request->pay_until,
            'user' => $admin ? $admin->emp_id : 0,
            'flag' => $request->flag ?? '0'
        ]);


        // Legacy Audit Logging (as requested)
        HrAudit::create([
            'emp_id' => $client->member_no, // Member No matches emp_id in examples
            'type' => 'Print Order',
            'reference_id' => $order->id,
            'snapshot' => [
                'id' => $order->id,
                'member_no' => $client->member_no,
                'client_name' => $client->arabic_name,
                'amount' => $request->amount,
                'bank' => $request->bank,
                'inst_type' => $request->inst_type,
                'pay_until' => $request->pay_until,
                'issue_date' => $order->issue_date
            ],
            'reg_by' => $admin ? $admin->id : 1,
            'created_at' => now()
        ]);

        // General Audit Logging
        AuditLog::create([
            'action' => 'print_payment_order',
            'model' => PayOrder::class,
            'model_id' => $order->id,
            'details' => json_encode([
                'client' => $client->arabic_name,
                'member_no' => $client->member_no,
                'amount' => $request->amount,
                'project' => $request->prj_id,
                'type' => $request->inst_type
            ]),
            'user_id' => $admin ? $admin->id : null,
            'ip_address' => $request->ip()
        ]);

        return response()->json([
            'success' => true,
            'order_id' => $order->id
        ]);
    }

    /**
     * Render the printable payment order
     */
    public function show($id)
    {
        $order = PayOrder::with(['project'])->findOrFail($id);
        $client = Client::where('member_no', $order->member_no)->firstOrFail();

        // Get unit details for the order
        $reservation = DB::table('default_reservations as r')
            ->leftJoin('flats as f', 'f.flt_id', '=', 'r.flt_id')
            ->where('r.cl_id', $client->member_no)
            ->where('r.prj_id', $order->prj_id)
            ->select('r.*', 'f.txt as flat_txt')
            ->first();

        $bank_parts = explode('|', $order->bank);
        $bank_name = $bank_parts[0] ?? '';
        $bank_account = $bank_parts[1] ?? '';

        $amount_in_words = $this->arabicNumberService->moneyToArabicWords($order->amount);

        return view('admin.clients.print.payment_order', compact(
            'order',
            'client',
            'reservation',
            'bank_name',
            'bank_account',
            'amount_in_words'
        ));
    }
}
