<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class LettersController extends Controller
{
    /**
     * Display warning letters page
     */
    public function printWarnings(Request $request)
    {
        $projectId = $request->input('project_id', 0);
        $area = $request->input('area', 0);

        // Fetch projects
        $projects = DB::table('projects')->get();

        // Fetch areas for selected project
        $areas = [];
        if ($projectId > 0) {
            $areas = DB::table('default_temp_warnings')
                ->where('prj_id', $projectId)
                ->select('area')
                ->distinct()
                ->orderBy('area')
                ->get();
        }

        // Fetch clients with warnings from temp table
        $clients = [];
        if ($projectId > 0) {
            $query = DB::table('default_temp_warnings')
                ->where('prj_id', $projectId)
                ->select('name', 'mem_no', 'area', 'bldg', 'flat_num');

            if ($area != 0 && $area != '0') {
                $decodedArea = base64_decode(str_replace('xx', '=', $area));
                $query->where('area', $decodedArea);
            }

            $clients = $query->get();
        }

        // Get clients with garden assignments (optional - table may not exist)
        $gardenClients = [];
        if ($projectId > 0) {
            try {
                $gardenClients = DB::table('default_garden_resv')
                    ->where('prj_id', $projectId)
                    ->pluck('mem_no')
                    ->toArray();
            } catch (\Exception $e) {
                // Table doesn't exist, garden clients will be empty
                $gardenClients = [];
            }
        }

        return view('admin.letters.print_warnings', compact('projects', 'areas', 'clients', 'projectId', 'area', 'gardenClients'));
    }

    /**
     * Generate warning letter HTML
     */
    public function generateLetter(Request $request)
    {
        $memberNos = explode("','", $request->input('mem_no'));
        $projectId = $request->input('prj');
        $date = $request->input('date');
        $number = $request->input('num');
        $type = $request->input('type', 'standard'); // standard, garden, info

        $html = '';

        foreach ($memberNos as $memberNo) {
            if (empty($memberNo))
                continue;

            // Fetch client data
            $client = DB::table('default_clients as c')
                ->leftJoin('flats as f', 'c.flat_id', '=', 'f.flat_id')
                ->leftJoin('projects as p', 'f.prj_id', '=', 'p.prj_id')
                ->where('c.memb_no', $memberNo)
                ->select('c.name', 'c.memb_no', 'f.area', 'f.bldg', 'f.flat_num', 'p.name as project_name')
                ->first();

            if (!$client)
                continue;

            // Generate letter based on type
            if ($type == 'garden') {
                $html .= $this->generateGardenLetter($client, $date, $number);
            } elseif ($type == 'info') {
                $html .= $this->generateInfoLetter($client, $date, $number);
            } else {
                $html .= $this->generateStandardLetter($client, $date, $number);
            }
        }

        return response($html);
    }

    /**
     * Generate standard warning letter
     */
    private function generateStandardLetter($client, $date, $number)
    {
        return view('admin.letters.templates.standard', compact('client', 'date', 'number'))->render();
    }

    /**
     * Generate garden warning letter
     */
    private function generateGardenLetter($client, $date, $number)
    {
        return view('admin.letters.templates.garden', compact('client', 'date', 'number'))->render();
    }

    /**
     * Generate info letter
     */
    private function generateInfoLetter($client, $date, $number)
    {
        return view('admin.letters.templates.info', compact('client', 'date', 'number'))->render();
    }

    /**
     * Delete project temporary data
     */
    public function deleteProjectData(Request $request)
    {
        $projectId = $request->input('prj');

        DB::table('default_temp_warnings')->where('prj_id', $projectId)->delete();

        return response()->json(['success' => true, 'message' => 'تم مسح البيانات بنجاح']);
    }

    /**
     * Review project data
     */
    public function reviewProject(Request $request)
    {
        $projectId = $request->input('prj');

        $data = DB::table('default_temp_warnings')
            ->where('prj_id', $projectId)
            ->select('name', 'mem_no', 'area', 'bldg', 'flat_num')
            ->get();

        return view('admin.letters.review', compact('data'))->render();
    }
}
