<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Project;
use App\Models\Flat;
use App\Models\InstallmentTable;
use App\Models\Job;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Log;
use App\Models\AuditLog;
use Illuminate\Support\Facades\Auth;

class InventoryController extends Controller
{
    public function projects()
    {
        $projects = Project::where('flag', 'Y')->orderBy('prj_id', 'asc')->get();

        // Count reservations per project
        $counts = DB::table('default_reservations as r')
            ->join('default_clients as c', 'c.member_no', '=', 'r.cl_id')
            ->where('c.flag', 'Y')
            ->whereNotNull('r.prj_id')
            ->groupBy('r.prj_id')
            ->select('r.prj_id', DB::raw('count(r.cl_id) as cnt'))
            ->pluck('cnt', 'prj_id')
            ->toArray();

        return view('admin.inventory.projects', compact('projects', 'counts'));
    }

    public function flats($projectId)
    {
        $project = Project::findOrFail($projectId);
        $flats = Flat::where('prj_id', $projectId)->where('flag', 'Y')->orderBy('flt_id', 'desc')->get();
        return view('admin.inventory.flats', compact('project', 'flats'));
    }

    public function installments($projectId, $flatId)
    {
        $project = Project::findOrFail($projectId);
        $flat = Flat::where('prj_id', $projectId)->where('flt_id', $flatId)->firstOrFail();

        $installments = InstallmentTable::where('project', $projectId)
            ->where('flat_type', $flatId)
            ->orderBy('bldg_no')
            ->orderBy('flat_num')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        // Calculate dynamic total
        $installments->transform(function ($item) {
            $item->total = $item->inst_amount + ($item->addon_amount ?? 0);
            return $item;
        });

        // Group by Building and Flat Number
        $groupedInstallments = $installments->groupBy(function ($item) {
            return ($item->bldg_no ?? 'N/A') . ' - ' . ($item->flat_num ?? 'N/A');
        });

        // return view('admin.inventory.installments', compact('project', 'flat', 'groupedInstallments', 'installments'));
        return view('admin.inventory.installments', compact('project', 'flat', 'groupedInstallments', 'installments'));
    }

    public function downloadTemplate()
    {
        // Clear all previous output buffers to prevent file corruption (Chrome fix)
        while (ob_get_level()) {
            ob_end_clean();
        }

        $path = public_path('installments_template.xls');

        return response()->download($path, 'installments_template.xls', [
            'Content-Type' => 'application/vnd.ms-excel',
        ]);
    }

    public function importInstallments(Request $request, $projectId, $flatId)
    {
        $request->validate([
            'csv_file' => 'required|file|mimes:csv,txt',
        ]);

        $file = $request->file('csv_file');

        DB::beginTransaction();
        try {
            // Buffer rows before deleting
            $rowsToInsert = [];
            if (($handle = fopen($file->getRealPath(), 'r')) !== false) {
                $ext = strtolower($file->getClientOriginalExtension());

                // Parse XML/XLS (SpreadsheetML)
                if (in_array($ext, ['xml', 'xls'])) {
                    $content = file_get_contents($file->getRealPath());
                    $xml = simplexml_load_string($content);

                    $ss_ns = 'urn:schemas-microsoft-com:office:spreadsheet';
                    $ws_nodes = $xml->children($ss_ns);

                    // Iterate rows (skip first row as header)
                    $isHeader = true;
                    if (isset($ws_nodes->Worksheet)) {
                        foreach ($ws_nodes->Worksheet->Table->Row as $row) {
                            if ($isHeader) {
                                $isHeader = false;
                                continue;
                            }

                            $cells = [];
                            foreach ($row->Cell as $cell) {
                                $cells[] = (string) $cell->Data;
                            }

                            // Ensure row has enough columns
                            if (count($cells) < 5)
                                continue;

                            $rowsToInsert[] = [
                                'project' => $projectId,
                                'flat_type' => $flatId,
                                'bldg_no' => trim($cells[0]),
                                'flat_num' => trim($cells[1]),
                                'year' => trim($cells[2]),
                                'month' => trim($cells[3]),
                                'inst_amount' => trim($cells[4]),
                                'addon_amount' => isset($cells[5]) ? trim($cells[5]) : 0,
                                'total' => trim($cells[4]) + (isset($cells[5]) ? trim($cells[5]) : 0),
                                'flag' => 'Y'
                            ];
                        }
                    }
                }
                // Parse CSV (Default)
                else {
                    $header = fgetcsv($handle); // Skip header
                    while (($row = fgetcsv($handle)) !== false) {
                        if (count($row) < 5)
                            continue; // Skip invalid rows

                        $rowsToInsert[] = [
                            'project' => $projectId,
                            'flat_type' => $flatId,
                            'bldg_no' => trim($row[0]),
                            'flat_num' => trim($row[1]),
                            'year' => trim($row[2]),
                            'month' => trim($row[3]),
                            'inst_amount' => trim($row[4]),
                            'addon_amount' => isset($row[5]) ? trim($row[5]) : 0,
                            'total' => trim($row[4]) + (isset($row[5]) ? trim($row[5]) : 0),
                            'flag' => 'Y'
                        ];
                    }
                }
                fclose($handle);
            }

            if (empty($rowsToInsert)) {
                DB::rollBack();
                return back()->with('error', 'No valid rows found in the CSV file. Previous data was NOT deleted.');
            }

            // Group rows by unit to perform targeted replacement
            $groupedByUnit = [];
            foreach ($rowsToInsert as $row) {
                $unitKey = $row['bldg_no'] . '|' . $row['flat_num'];
                $groupedByUnit[$unitKey][] = $row;
            }

            $affectedUnitsCount = 0;
            foreach ($groupedByUnit as $unitKey => $unitRows) {
                list($bldg, $num) = explode('|', $unitKey);

                // Find the corresponding flat in this project
                $targetFlat = Flat::where('prj_id', $projectId)
                    ->where('bldg_id', $bldg)
                    ->where('flat_num', $num)
                    ->first();

                if ($targetFlat) {
                    // Delete existing rows ONLY for this specific unit
                    InstallmentTable::where('project', $projectId)
                        ->where('flat_type', $targetFlat->flt_id)
                        ->delete();

                    // Insert new rows
                    foreach ($unitRows as $data) {
                        $data['flat_type'] = $targetFlat->flt_id; // Ensure ID matches physical lookup
                        InstallmentTable::create($data);
                    }
                    $affectedUnitsCount++;
                }
            }

            DB::commit();

            // Log Audit
            $admin = Auth::guard('admin')->user();
            AuditLog::create([
                'action' => 'import_installments',
                'model' => InstallmentTable::class,
                'details' => json_encode([
                    'project_id' => $projectId,
                    'affected_units' => $affectedUnitsCount,
                    'total_rows' => count($rowsToInsert),
                ]),
                'user_id' => $admin ? $admin->id : null,
                'ip_address' => $request->ip()
            ]);

            return back()->with('success', "Import complete. Replaced data for $affectedUnitsCount units.");

            // Legacy/Unused code block below effectively removed by this replacement return logic
            // (keeping the structure valid for the closing braces outside this block)
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error importing file: ' . $e->getMessage());
        }
    }



    public function jobs()
    {
        $jobs = Job::all();
        return view('admin.inventory.jobs', compact('jobs'));
    }

    public function storeJob(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'desc' => 'nullable',
        ]);

        Job::create($request->all());
        return back()->with('success', 'Job added successfully.');
    }

    public function deleteJob($id)
    {
        Job::findOrFail($id)->delete();
        return back()->with('success', 'Job deleted successfully.');
    }
}
