<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class HrController extends Controller
{
    public function index()
    {
        // Migrating logic from Msabry::Get_Employees
        // Query: Select employees with their latest salary details.

        /*
            Original Query Concept:
            SELECT distinct e.*, a.*, s.*
            FROM default_employee e
            LEFT JOIN default_admins a ON e.emp_id = a.emp_id
            LEFT JOIN default_salary s ON s.emp_id = e.emp_id
            WHERE s.month = (MAX month) AND s.id = (MAX id) ...
        */

        // Simplified Eloquent Approach:
        // Get active employees, order by emp_order
        // We can load latest salary via a relationship or just fetch everything for now.

        $employees = \App\Models\Employee::with([
            'salaries' => function ($q) {
                $q->orderBy('id', 'desc')->limit(1);
            },
            'attendance' => function ($q) {
                $q->orderBy('id', 'desc')->limit(1);
            }
        ])
            ->where('flag', 'Y')
            ->where('emp_id', '!=', 1)
            ->where('job', '!=', 'مدير الموقع')
            ->orderBy('emp_order', 'asc')
            ->get();

        return view('admin.hr.index', compact('employees'));
    }

    public function create()
    {
        // Get generic job places for dropdown
        $jobPlaces = \App\Models\Employee::where('flag', 'Y')
            ->select('job_place')
            ->distinct()
            ->pluck('job_place');

        $managers = \App\Models\Employee::where('flag', 'Y')
            ->orderBy('arabic_name', 'asc')
            ->get();

        return view('admin.hr.create', compact('jobPlaces', 'managers'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'arabic_name' => 'required',
            'english_name' => 'required',
            'nat_id' => 'required|numeric',
            'mobile' => 'nullable',
            'job' => 'required',
            'job_place' => 'required',
        ]);

        $data = $request->except('_token');

        // Add defaults
        $data['flag'] = 'Y';
        // Legacy 'emp_order' default
        $maxOrder = \App\Models\Employee::max('emp_order');
        $data['emp_order'] = $maxOrder + 1;

        \App\Models\Employee::create($data);

        return redirect()->route('hr.index')->with('success', 'Employee created successfully.');
    }

    public function edit($id)
    {
        $employee = \App\Models\Employee::findOrFail($id);

        $jobPlaces = \App\Models\Employee::where('flag', 'Y')
            ->select('job_place')
            ->distinct()
            ->pluck('job_place');

        // Managers list (all active employees)
        $managers = \App\Models\Employee::where('flag', 'Y')
            ->orderBy('arabic_name', 'asc')
            ->get();

        return view('admin.hr.edit', compact('employee', 'jobPlaces', 'managers'));
    }

    public function update(Request $request, $id)
    {
        $employee = \App\Models\Employee::findOrFail($id);

        $request->validate([
            'arabic_name' => 'required',
            'english_name' => 'required',
            'nat_id' => 'required|numeric',
            'mobile' => 'nullable',
            'job' => 'required',
            'job_place' => 'required',
        ]);

        $data = $request->except(['_token', '_method']);

        $employee->update($data);

        return redirect()->route('hr.index')->with('success', 'Employee updated successfully.');
    }

    public function destroy($id)
    {
        $employee = \App\Models\Employee::findOrFail($id);
        // Legacy uses soft delete via flag
        $employee->update(['flag' => 'N']);

        return redirect()->route('hr.index')->with('success', 'Employee deactivated.');
    }

    public function salaries(Request $request)
    {
        $month = $request->get('month', date('m'));
        $jobPlace = $request->get('job_place');

        $query = \App\Models\Employee::with([
            'salaries' => function ($q) use ($month) {
                $q->where('month', $month)->orderBy('id', 'desc');
            }
        ])->where('flag', 'Y')
            ->where('job', '!=', 'مدير الموقع');

        if ($jobPlace) {
            $query->where('job_place', $jobPlace);
        }

        $employees = $query->get();

        $jobPlaces = \App\Models\Employee::where('flag', 'Y')
            ->select('job_place')
            ->distinct()
            ->pluck('job_place');

        return view('admin.hr.salaries', compact('employees', 'month', 'jobPlaces', 'jobPlace'));
    }

    public function attendance(Request $request)
    {
        $from = $request->get('from', date('Y-m-01'));
        $to = $request->get('to', date('Y-m-d'));

        $attendance = \App\Models\Attendance::whereBetween('day', [$from, $to])
            ->orderBy('day', 'asc')
            ->get();

        return view('admin.hr.attendance', compact('attendance', 'from', 'to'));
    }

    public function bonuses(Request $request)
    {
        $year = $request->get('year', date('Y'));
        $jobPlace = $request->get('job_place');

        $query = \App\Models\Employee::with([
            'bonuses' => function ($q) use ($year) {
                $q->where('year', $year);
            }
        ])->where('flag', 'Y')
            ->where('job', '!=', 'مدير الموقع');

        if ($jobPlace) {
            $query->where('job_place', $jobPlace);
        }

        $employees = $query->get();

        $jobPlaces = \App\Models\Employee::where('flag', 'Y')
            ->select('job_place')
            ->distinct()
            ->pluck('job_place');

        return view('admin.hr.bonuses', compact('employees', 'year', 'jobPlaces', 'jobPlace'));
    }

    public function vacations()
    {
        $vacations = \App\Models\Vacation::with('employee')
            ->orderBy('date_from', 'desc')
            ->get();

        return view('admin.hr.vacations', compact('vacations'));
    }

    public function updateSalaries(Request $request)
    {
        $salaries = $request->input('salaries', []);
        $month = $request->input('month', date('m'));

        foreach ($salaries as $emp_id => $data) {
            // Update Base Salary in Employee model
            $employee = \App\Models\Employee::find($emp_id);
            if ($employee) {
                $employee->update(['salary' => $data['base_salary'] ?? 0]);
            }

            // Update or Create Salary record for the month
            \App\Models\Salary::updateOrCreate(
                ['emp_id' => $emp_id, 'month' => $month],
                [
                    'incentive' => $data['incentive'] ?? 0,
                    'meal' => $data['meal'] ?? 0,
                    'loans' => $data['loans'] ?? 0,
                    'absence' => $data['absence'] ?? 0,
                    // Keeping other fields if necessary or setting defaults
                    'extra' => $data['extra'] ?? 0,
                    'transp' => $data['transp'] ?? 0,
                    'reg_by' => auth('admin')->id(),
                ]
            );
        }

        return redirect()->route('hr.salaries', ['month' => $month])->with('success', 'Salaries updated successfully.');
    }

    public function printSalary(Request $request, $id)
    {
        $month = $request->get('month', date('m'));
        $employee = \App\Models\Employee::with([
            'salaries' => function ($q) use ($month) {
                $q->where('month', $month)->orderBy('id', 'desc');
            }
        ])->findOrFail($id);

        $salary = $employee->salaries->first();
        if ($salary) {
            $this->saveAudit($employee->emp_id, 'salary', $salary->id, [
                'employee' => $employee->arabic_name,
                'month' => $month,
                'year' => date('Y'),
                'base_salary' => $employee->salary,
                'incentive' => $salary->incentive,
                'meal' => $salary->meal,
                'extra' => $salary->extra,
                'transp' => $salary->transp,
                'absence' => $salary->absence,
                'loans' => $salary->loans,
                'total' => ($employee->salary + $salary->incentive + $salary->meal + $salary->extra + $salary->transp) - ($salary->absence + $salary->loans)
            ]);
        }

        $options = $request->only(['sign_x', 'pr_x', 'emp_x']);

        return view('admin.hr.print_salary', compact('employee', 'month', 'options'));
    }

    public function printAllSalaries(Request $request)
    {
        $month = $request->get('month', date('m'));
        $jobPlace = $request->get('job_place');

        $query = \App\Models\Employee::with([
            'salaries' => function ($q) use ($month) {
                $q->where('month', $month)->orderBy('id', 'desc');
            }
        ])->where('flag', 'Y')
            ->where('job', '!=', 'مدير الموقع');

        if ($jobPlace) {
            $query->where('job_place', $jobPlace);
        }

        $employees = $query->get();

        foreach ($employees as $emp) {
            $salary = $emp->salaries->first();
            if ($salary) {
                $this->saveAudit($emp->emp_id, 'salary', $salary->id, [
                    'employee' => $emp->arabic_name,
                    'month' => $month,
                    'year' => date('Y'),
                    'base_salary' => $emp->salary,
                    'incentive' => $salary->incentive,
                    'meal' => $salary->meal,
                    'extra' => $salary->extra,
                    'transp' => $salary->transp,
                    'absence' => $salary->absence,
                    'loans' => $salary->loans,
                    'total' => ($emp->salary + $salary->incentive + $salary->meal + $salary->extra + $salary->transp) - ($salary->absence + $salary->loans)
                ]);
            }
        }

        $options = $request->only(['sign_x', 'pr_x', 'emp_x']);

        return view('admin.hr.print_all_salaries', compact('employees', 'month', 'jobPlace', 'options'));
    }

    public function printBonus(Request $request, $id)
    {
        $bonus = \App\Models\Bonus::with('employee')->findOrFail($id);

        $this->saveAudit($bonus->emp_id, 'bonus', $bonus->id, [
            'employee' => $bonus->employee->arabic_name,
            'year' => $bonus->year,
            'incentive' => $bonus->incentive,
            'meal' => $bonus->meal,
            'loans' => $bonus->loans,
            'extra' => $bonus->amount,
            'transp' => $bonus->transp,
            'taswia' => $bonus->taswia,
            'total' => ($bonus->employee->salary + $bonus->incentive + $bonus->meal + $bonus->amount + $bonus->transp + $bonus->taswia) - $bonus->loans,
            'date' => date('Y-m-d', $bonus->crdate)
        ]);

        $options = $request->only(['sign_x', 'pr_x', 'emp_x']);

        return view('admin.hr.print_bonus', compact('bonus', 'options'));
    }

    public function printAllBonuses(Request $request)
    {
        $year = $request->get('year', date('Y'));
        $jobPlace = $request->get('job_place');

        $query = \App\Models\Bonus::with('employee')
            ->where('year', $year)
            ->whereHas('employee', function ($q) {
                $q->where('job', '!=', 'مدير الموقع');
            });

        if ($jobPlace) {
            $query->whereHas('employee', function ($q) use ($jobPlace) {
                $q->where('job_place', $jobPlace);
            });
        }

        $bonuses = $query->get();

        foreach ($bonuses as $bonus) {
            $this->saveAudit($bonus->emp_id, 'bonus', $bonus->id, [
                'employee' => $bonus->employee->arabic_name,
                'year' => $bonus->year,
                'incentive' => $bonus->incentive,
                'meal' => $bonus->meal,
                'loans' => $bonus->loans,
                'extra' => $bonus->amount,
                'transp' => $bonus->transp,
                'taswia' => $bonus->taswia,
                'total' => ($bonus->employee->salary + $bonus->incentive + $bonus->meal + $bonus->amount + $bonus->transp + $bonus->taswia) - $bonus->loans,
                'date' => date('Y-m-d', $bonus->crdate)
            ]);
        }

        $options = $request->only(['sign_x', 'pr_x', 'emp_x']);

        return view('admin.hr.print_all_bonuses', compact('bonuses', 'year', 'jobPlace', 'options'));
    }

    public function audit()
    {
        $logs = \App\Models\HrAudit::with('employee')->orderBy('created_at', 'desc')->paginate(50);
        return view('admin.hr.audit', compact('logs'));
    }

    public function updateBonuses(Request $request)
    {
        $year = $request->get('year', date('Y'));
        $bonusesData = $request->get('bonuses', []);

        foreach ($bonusesData as $empId => $data) {
            // Check if any field has a value
            // 'extra' is the legacy 'amount' field in our Bonus model
            if (isset($data['incentive']) || isset($data['meal']) || isset($data['loans']) || isset($data['extra']) || isset($data['transp']) || isset($data['taswia'])) {
                \App\Models\Bonus::updateOrCreate(
                    [
                        'emp_id' => $empId,
                        'year' => $year,
                    ],
                    [
                        'incentive' => $data['incentive'] ?? 0,
                        'meal' => $data['meal'] ?? 0,
                        'loans' => $data['loans'] ?? 0,
                        'amount' => $data['extra'] ?? 0,
                        'transp' => $data['transp'] ?? 0,
                        'taswia' => $data['taswia'] ?? 0,
                        'crdate' => time()
                    ]
                );
            }
        }

        return redirect()->route('hr.bonuses', ['year' => $year])->with('success', 'Bonuses/Additions updated successfully.');
    }

    private function saveAudit($empId, $type, $refId, $snapshot)
    {
        \App\Models\HrAudit::create([
            'emp_id' => $empId,
            'type' => $type,
            'reference_id' => $refId,
            'snapshot' => $snapshot,
            'reg_by' => auth()->id() ?? 1
        ]);
    }
}
