<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Client;
use App\Models\Project;
use App\Models\User;
use App\Models\Reservation;
use App\Models\Receipt;
use App\Models\Flat;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class DashboardController extends Controller
{
    public function index()
    {
        // 1. Counters
        $users_count = User::count();
        $clients_count = Client::where('flag', 'Y')->count();
        $projects_count = Project::count();

        // Updates count (Live Data)
        $updates_count = 0;
        if (DB::getSchemaBuilder()->hasTable('default_clients_live')) {
            $updates_count = DB::table('default_clients_live')->count();
        }

        // 2. Financials (Total Deposits) per Project
        $deposits = [];
        $deposits[1] = Receipt::where('prj_id', 1)->where('flag', 'Y')->sum('value');
        $deposits[2] = Receipt::where('prj_id', 2)->where('flag', 'Y')->sum('value');
        $deposits[3] = Receipt::where('prj_id', 3)->where('flag', 'Y')->sum('value');

        $total_deposits = array_sum($deposits);

        // 3. Allocation (Reservations) per Project
        $allocations = [];
        for ($i = 1; $i <= 5; $i++) {
            $allocations[$i] = Reservation::where('prj_id', $i)->where('flat_num', '!=', '')->whereNotNull('flat_num')->count();
        }

        // 4. Unallocated
        $unallocated = [];
        for ($i = 1; $i <= 5; $i++) {
            $unallocated[$i] = Reservation::where('prj_id', $i)->where(function ($q) {
                $q->whereNull('flat_num')->orWhere('flat_num', '');
            })->count();
        }

        return view('admin.dashboard.index', compact(
            'users_count',
            'clients_count',
            'projects_count',
            'updates_count',
            'deposits',
            'total_deposits',
            'allocations',
            'unallocated'
        ));
    }

    public function visual($id)
    {
        $project = Project::find($id);
        if (!$project)
            abort(404);

        $reservations = Reservation::where('default_reservations.prj_id', $id)
            ->leftJoin('default_clients', 'default_reservations.cl_id', '=', 'default_clients.member_no')
            ->leftJoin('flats', 'default_reservations.flt_id', '=', 'flats.flt_id')
            ->select(
                'default_reservations.*',
                'default_clients.cl_id as client_id',
                'default_clients.member_no as client_member_no',
                'default_clients.arabic_name',
                'default_clients.onoff',
                'flats.area as flat_area'
            )
            ->get();

        $reservedMap = [];
        $bldgAreaMap = [];

        foreach ($reservations as $res) {
            $bldg = (string) ($res->bldg_id ?: '0');
            $unitNum = (string) $res->flat_num;
            $reservedMap[$bldg][$unitNum] = $res;

            if ($bldg !== '0' && $bldg !== '' && $res->flat_area) {
                if (!isset($bldgAreaMap[$res->flat_area])) {
                    $bldgAreaMap[$res->flat_area] = [];
                }
                if (!in_array($bldg, $bldgAreaMap[$res->flat_area])) {
                    $bldgAreaMap[$res->flat_area][] = $bldg;
                }
            }
        }

        // Fetch Duplexes
        $duplix = DB::table('default_duplix')->where('prj_id', $id)->get();
        $duplix_arr = [];
        $all_duplix_arr = [];
        foreach ($duplix as $d) {
            $duplix_arr[] = $d->bldg . "-" . $d->flat_no1 . "|" . $d->bldg . "-" . $d->flat_no2;
            $all_duplix_arr[] = $d->bldg . "-" . $d->flat_no1;
            $all_duplix_arr[] = $d->bldg . "-" . $d->flat_no2;
        }

        $groupings = [];
        $levels = ($id == 2) ? 5 : (($id == 1) ? 25 : 7);
        $inc_counter = ($id == 1) ? 4 : 1;

        // Fetch all area types from flats table for this project
        $areaTypes = DB::table('flats')->where('prj_id', $id)->select('area', 'txt')->distinct()->get();

        foreach ($areaTypes as $type) {
            $area = $type->area;
            $areaLabel = $type->txt ?: $area;

            if ($area == 'sepvila') {
                $rangeEnd = ($id == 1) ? 16 : 62;
                for ($i = 1; $i <= $rangeEnd; $i++) {
                    $num = (string) $i;
                    $res = $reservedMap['0'][$num] ?? null;
                    $groupings[$areaLabel]['Villas'][$num] = $this->createUnitData($num, $res, '0', 0);
                }
            } elseif ($area == 'twinvila' && $id == 3) {
                for ($i = 63; $i <= 122; $i++) {
                    foreach (['A', 'B'] as $pref) {
                        $num = $pref . $i;
                        $res = $reservedMap['0'][$num] ?? null;
                        $groupings[$areaLabel]['Villas'][$num] = $this->createUnitData($num, $res, '0', 0);
                    }
                }
            } elseif ($area == 'grand_265' && $id == 3) {
                // Hardcoded range for grand_265 if needed, or just from reservations
                if (isset($bldgAreaMap[$area])) {
                    foreach ($bldgAreaMap[$area] as $b) {
                        $res = $reservedMap[$b][''] ?? null;
                        $groupings[$areaLabel]['Villas'][$b] = $this->createUnitData($b, $res, $b, 0);
                    }
                }
            } else {
                // Apartment areas or other buildings
                if (isset($bldgAreaMap[$area])) {
                    $currentLevels = (in_array($area, ['itavila', 'itavila2'])) ? 12 : $levels;
                    foreach ($bldgAreaMap[$area] as $b) {
                        for ($f = 0; $f <= $currentLevels; $f += $inc_counter) {
                            for ($u = 1; $u <= 4; $u++) {
                                // Only Project 2 / Area 150 uses A/B prefixes for apartments in legacy
                                $prefixes = ($id == 2 && $area == '150') ? ['A', 'B'] : [''];
                                foreach ($prefixes as $pref) {
                                    $num = ($id == 1) ? (string) ($f + $u) : $pref . ($f * 100 + $u);
                                    // Logic for available/reserved
                                    $res = $reservedMap[$b][$num] ?? null;

                                    // For Project 1, we only show if it was actually reserved or within a building we track
                                    if ($id != 1 || isset($reservedMap[$b][$num])) {
                                        $groupings[$areaLabel]['Bldg ' . $b][$num] = $this->createUnitData($num, $res, $b, $f);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        // Final Sorting: natural sort for unit numbers and building keys
        foreach ($groupings as $area => &$buildings) {
            uksort($buildings, 'strnatcmp');
            foreach ($buildings as $bLabel => &$units) {
                uksort($units, 'strnatcmp');
            }
        }

        $clients = Client::where('flag', 'Y')->select('cl_id', 'member_no', 'arabic_name')->orderBy('arabic_name')->get();

        return view('admin.dashboard.visual', compact('project', 'groupings', 'clients', 'duplix_arr', 'all_duplix_arr'));
    }

    private function createUnitData($num, $reservation, $bldg, $floor)
    {
        return (object) [
            'flat_num' => $num,
            'bldg_id' => $bldg,
            'floor' => $floor,
            'is_reserved' => !is_null($reservation),
            'client_id' => $reservation ? $reservation->client_id : null,
            'client_member_no' => $reservation ? $reservation->client_member_no : null,
            'arabic_name' => $reservation ? $reservation->arabic_name : null,
            'sess' => $reservation ? $reservation->sess : null,
            'subsess' => $reservation ? $reservation->subsess : null,
            'res_kind' => $reservation ? $reservation->res_kind : null,
            'onoff' => $reservation ? $reservation->onoff : null,
        ];
    }
}
