<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Models\Client;
use App\Models\Reservation;
use App\Models\Receipt;
use App\Models\Project;
use App\Models\Bank;
use App\Models\Flat;
use App\Models\TotalPrice;
use App\Models\HrAudit;

class ClientsController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->get('search');
        $prj = $request->get('prj', 0);

        $query = Client::query()
            ->leftJoin('default_reservations as r', 'r.cl_id', '=', 'default_clients.member_no')
            ->leftJoin('projects as p', 'p.prj_id', '=', 'r.prj_id')
            ->leftJoin('flats as f', function ($join) {
                $join->on('f.flt_id', '=', 'r.flt_id')
                    ->on('f.prj_id', '=', 'r.prj_id');
            })
            ->where('default_clients.flag', 'Y')
            ->select(
                'default_clients.*',
                'p.name as project_name',
                'r.bldg_id',
                'r.flat_num',
                'r.flt_order',
                'r.sess',
                'r.subsess',
                'r.prj_id',
                'f.txt as flat_txt'
            )
            ->distinct();

        if ($prj != 0 && $prj != 9999) {
            $query->where('r.prj_id', $prj);
        }

        if ($search) {
            $search2 = $this->handleArabicVariations($search);
            $query->where(function ($q) use ($search, $search2) {
                $q->where('default_clients.member_no', 'like', "%{$search}%")
                    ->orWhere('default_clients.old_member_no', 'like', "%{$search}%")
                    ->orWhere('default_clients.file_no', 'like', "%{$search}%")
                    ->orWhere('default_clients.arabic_name', 'like', "%{$search}%")
                    ->orWhere('default_clients.arabic_name', 'like', "%{$search2}%")
                    ->orWhere('default_clients.english_name', 'like', "%{$search}%")
                    ->orWhere('default_clients.mobile', 'like', "%{$search}%");
            });
        }

        $clients = $query->orderBy('default_clients.cl_id', 'desc')->paginate(25);
        $projects = Project::where('flag', 'Y')->get();

        // Data for "Add Client" modal
        $next_old_member_no = Client::max('old_member_no') + 1;
        $next_member_no = Client::max('member_no') + 1;
        $next_file_no = Client::max('file_no') + 1;
        $states = DB::table('states')->orderBy('name')->get();

        return view('admin.clients.index', compact(
            'clients',
            'projects',
            'next_old_member_no',
            'next_member_no',
            'next_file_no',
            'states'
        ));
    }

    public function ajaxSearch(Request $request)
    {
        $search = trim($request->get('term'));

        $clients = Client::where('old_member_no', '=', $search)
            ->orWhere('old_member_no', 'like', "%{$search}%")
            ->orWhere('file_no', 'like', "%{$search}%")
            ->orWhere('arabic_name', 'like', "%{$search}%")
            ->select('cl_id', 'member_no', 'old_member_no', 'arabic_name')
            ->orderByRaw("CASE WHEN old_member_no = ? THEN 1 ELSE 2 END", [$search])
            ->limit(50)
            ->get();

        return response()->json($clients);
    }

    public function checkExistence(Request $request)
    {
        $name = $request->get('name');
        $exists = Client::where('arabic_name', $name)->exists();
        return response()->json(['exists' => $exists]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'arabic_name' => 'required|string|max:255',
            'english_name' => 'required|string|max:255',
            'old_member_no' => 'required',
            'mobile' => 'required',
            'nat_id' => 'required|digits:14',
        ]);

        // Generate member_no if not provided (legacy auto-generation logic)
        $member_no = $request->member_no;
        if (!$member_no) {
            $member_no = (int) $request->old_member_no + 100;
        }

        $client = Client::create([
            'arabic_name' => $request->arabic_name,
            'english_name' => $request->english_name,
            'old_member_no' => $request->old_member_no,
            'member_no' => $member_no,
            'member_start_date' => $request->member_start_date,
            'nat_id' => $request->nat_id,
            'milit_card_id' => $request->milit_card_id,
            'airforce_id' => $request->airforce_id,
            'nationality' => $request->nationality ?? 'مصرى',
            'address' => $request->address,
            'mobile' => $request->mobile,
            'gender' => $request->gender,
            'city' => $request->city,
            'state' => $request->state,
            'per_email' => $request->per_email,
            'teba_email' => $request->teba_email,
            'user_name' => $request->user_name,
            'bank_name' => $request->bank_name,
            'account_no' => $request->account_no,
            'file_no' => $request->file_no,
            'related_to' => $request->related_to,
            'relation' => $request->relation,
            'member_type' => $request->member_type,
            'move_status' => $request->move_status,
            'comment' => $request->comment,
            'reg_by' => Auth::id(),
            'flag' => 'Y',
        ]);

        // Audit Log
        HrAudit::create([
            'emp_id' => $client->member_no,
            'type' => 'add_client',
            'reference_id' => $client->cl_id,
            'snapshot' => $client->toArray(),
            'reg_by' => Auth::id(),
            'created_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => __('Client added successfully'),
            'cl_id' => $client->cl_id
        ]);
    }

    private function handleArabicVariations($str)
    {
        $search2 = $str;
        if (str_contains($str, 'عبدالله')) {
            $search2 = str_replace('عبدالله', 'عبد الله', $str);
        } elseif (str_contains($str, 'عبد الله')) {
            $search2 = str_replace('عبد الله', 'عبدالله', $str);
        } elseif (str_contains($str, 'ى')) {
            $search2 = str_replace('ى', 'ي', $str);
        } elseif (str_contains($str, 'ي')) {
            $search2 = str_replace('ي', 'ى', $str);
        } elseif (str_contains($str, 'ا')) {
            $search2 = str_replace('ا', 'أ', $str);
        } elseif (str_contains($str, 'أ')) {
            $search2 = str_replace('أ', 'ا', $str);
        } elseif (str_contains($str, 'ه')) {
            $search2 = str_replace('ه', 'ة', $str);
        }
        return $search2;
    }

    public function reservations($id)
    {
        $client = Client::where('cl_id', $id)->orWhere('member_no', $id)->firstOrFail();

        $reservations = DB::table('default_reservations as r')
            ->leftJoin('flats as f', function ($join) {
                $join->on('f.flt_id', '=', 'r.flt_id')
                    ->on('f.prj_id', '=', 'r.prj_id');
            })
            ->leftJoin('projects as p', 'p.prj_id', '=', 'r.prj_id')
            ->where('r.cl_id', $client->member_no)
            ->select('r.*', 'f.txt as flat_txt', 'p.name as project_name')
            ->get();

        if ($reservations->isEmpty()) {
            $reservations = DB::table('default_reservations as r')
                ->leftJoin('flats as f', function ($join) {
                    $join->on('f.flt_id', '=', 'r.flt_id')
                        ->on('f.prj_id', '=', 'r.prj_id');
                })
                ->leftJoin('projects as p', 'p.prj_id', '=', 'r.prj_id')
                ->where('r.cl_id', $client->cl_id)
                ->select('r.*', 'f.txt as flat_txt', 'p.name as project_name')
                ->get();
        }

        $data = [];
        $accountService = new \App\Services\ClientAccountService();

        foreach ($reservations as $res) {
            // Get accurate project-specific financial data
            $stats = $accountService->calculateAccountReport($client->cl_id, $res->prj_id);

            $pending = DB::table('default_pendings')
                ->where('cl_id', $client->member_no)
                ->where('prj_id', $res->prj_id)
                ->where('status', 'N')
                ->first();

            $data[] = [
                'res' => $res,
                'paid' => $stats['total'], // This is "Total Paid" excluding chk and specific types
                'required' => $stats['biggest_total'], // This is "Required Total" until current month
                'remaining' => $stats['remain'], // Accurate balance
                'pending' => $pending
            ];
        }

        $gardens = [];
        foreach ($data as $item) {
            if ($item['res']->garden == 'Y') {
                $gardens[] = $item['res'];
            }
        }

        $parking = DB::table('default_park_resv as p')
            ->leftJoin('projects as j', 'j.prj_id', '=', 'p.prj_id')
            ->where('p.cl_id', $client->member_no)
            ->select('p.*', 'j.name as project_name')
            ->get();

        return view('admin.clients.reservations', compact('client', 'data', 'parking', 'gardens'));
    }

    public function accountStatement($id)
    {
        $client = Client::where('cl_id', $id)->orWhere('member_no', $id)->firstOrFail();

        $sum = 0;
        $receipts = DB::table('default_receipts as r')
            ->join('default_clients as c', 'c.cl_id', '=', 'r.cl_id')
            ->where(function ($query) use ($id, $client) {
                $query->where('r.cl_id', $id)
                    ->orWhere('c.cl_id', $id)
                    ->orWhere('r.cl_id', $client->member_no);
            })
            ->where('r.flag', 'Y')
            ->where('c.flag', 'Y')
            ->where(function ($query) {
                $query->whereNotIn('r.type', ['100', '96', '99', '98'])
                    ->orWhereNull('r.type');
            })
            ->select('r.*')
            ->distinct()
            ->get();

        foreach ($receipts as $r) {
            $sum += $r->value;
        }

        $balances = DB::table('default_accounts')
            ->where('flag', 'Y')
            ->where('cl_id', $client->member_no)
            ->select('cl_id', DB::raw('sum(amount) as amount'))
            ->groupBy('cl_id')
            ->get();

        $banks = Bank::all();

        $reservations = DB::table('default_reservations as r')
            ->leftJoin('projects as p', 'p.prj_id', '=', 'r.prj_id')
            ->whereIn('r.cl_id', [$client->member_no, $client->cl_id])
            ->select('r.prj_id', 'p.name')
            ->get();

        return view('admin.clients.account_statement', compact(
            'client',
            'receipts',
            'sum',
            'balances',
            'banks',
            'reservations'
        ));
    }

    public function checks($id)
    {
        $client = Client::where('cl_id', $id)->orWhere('member_no', $id)->firstOrFail();

        $checks = DB::table('default_cheques')
            ->where('member_no', $client->old_member_no)
            ->orderBy('crdate', 'desc')
            ->get();

        $projects = [
            '1' => 'هليوبوليس',
            '2' => 'القاهره الجديده',
            '3' => 'عاصمه اداريه'
        ];

        return view('admin.clients.checks', compact('client', 'checks', 'projects'));
    }

    public function payments($id)
    {
        $client = Client::where('cl_id', $id)->orWhere('member_no', $id)->firstOrFail();

        // Get client's reservation data
        $reservation = DB::table('default_reservations as r')
            ->leftJoin('flats as f', function ($join) {
                $join->on('f.flt_id', '=', 'r.flt_id')
                    ->on('f.prj_id', '=', 'r.prj_id');
            })
            ->leftJoin('projects as p', 'p.prj_id', '=', 'r.prj_id')
            ->where('r.cl_id', $client->member_no)
            ->select('r.*', 'f.txt as flat_txt', 'p.name as project_name')
            ->first();

        if (!$reservation) {
            // No reservation: Initialize empty data and continue to view
            $receipts = collect();
            $totalPaid = 0;
            $checkTotal = 0;
            $parking = collect();
            $priceData = [];
            $subsess = '';
        } else {
            // Get all receipts for this client
            $receipts = DB::table('default_receipts as r')
                ->leftJoin('installment_type as t', 't.id', '=', 'r.type')
                ->where('r.cl_id', $client->cl_id)
                ->where('r.prj_id', $reservation->prj_id)
                ->where('r.flag', 'Y')
                ->whereNotIn('r.type', ['100', '96', '99', '98', '111', '7', '8', '101', '999'])
                ->whereNotNull('r.type')
                ->select('r.*', 't.type as type_txt')
                ->orderByRaw("STR_TO_DATE(r.date ,'%d-%m-%Y'), STR_TO_DATE(r.date ,'%m-%d-%Y') ASC")
                ->get();

            // Calculate totals
            $totalPaid = 0;
            $checkTotal = 0;
            foreach ($receipts as $receipt) {
                if ($receipt->chk != 'Y') {
                    $totalPaid += $receipt->value;
                }
                if ($receipt->chk == 'Y') {
                    $checkTotal += $receipt->value;
                }
            }

            // Get parking data (if table exists)
            $parking = collect();
            try {
                // Check if table exists first
                $tableExists = DB::select("SHOW TABLES LIKE 'default_park_resv'");

                if (!empty($tableExists)) {
                    $parking = DB::table('default_park_resv as p')
                        ->leftJoin('projects as j', 'j.prj_id', '=', 'p.prj_id')
                        ->where('p.cl_id', $client->member_no)
                        ->where('p.prj_id', $reservation->prj_id)
                        ->select('p.*', 'j.name as project_name')
                        ->get();
                }
            } catch (\Exception $e) {
                // Table doesn't exist, continue without parking data
                \Log::info("Parking table not found for client {$client->cl_id}: " . $e->getMessage());
            }

            // Calculate unit price and required payment using Cl_Account_rpt equivalent
            $priceData = $this->calculateClientAccount($client->cl_id, $reservation->prj_id);

            // Format subsess like legacy
            $subsess = '';
            if (isset($reservation->subsess) && $reservation->subsess == '1') {
                $subsess = 'متبقيات مرحله اولى';
            }
        }

        // Session names array (from legacy)
        $sessArr = [
            '1' => 'مرحله أولى',
            '11' => 'متبقيات مرحله اولى',
            '2' => 'مرحله ثانيه',
            '3' => 'مرحله ثالثه'
        ];

        return view('admin.clients.payments', compact(
            'client',
            'reservation',
            'receipts',
            'totalPaid',
            'checkTotal',
            'parking',
            'priceData',
            'sessArr',
            'subsess'
        ));
    }

    private function calculateClientAccount($cl_id, $prj_id)
    {
        $accountService = new \App\Services\ClientAccountService();
        return $accountService->calculateAccountReport($cl_id, $prj_id);
    }

    public function documents($id)
    {
        // Placeholder for Documents logic (Cl_files)
        return "Documents view for client $id";
    }

    public function editClient($id)
    {
        $client = Client::where('cl_id', $id)->orWhere('member_no', $id)->firstOrFail();
        $cities = DB::table('states')->get(); // Cities are in 'states' table in legacy
        $states = DB::table('cities')->get(); // States/Governorates are in 'cities' table in legacy
        $relations = DB::table('relation_type')->get();

        // Find existing relations if any
        $related_clients = Client::where('flag', 'Y')->orderBy('arabic_name')->get(['cl_id', 'arabic_name', 'old_member_no']);

        return view('admin.clients.edit', compact('client', 'cities', 'states', 'relations', 'related_clients'));
    }

    public function updateClient(Request $request, $id)
    {
        $client = Client::findOrFail($id);

        $request->validate([
            'arabic_name' => 'required|string|max:255',
            'english_name' => 'required|string|max:255',
            'nat_id' => 'required|digits:14',
        ]);

        $oldData = $client->toArray();
        $client->update($request->all());

        // Audit Log
        HrAudit::create([
            'emp_id' => Auth::id(), // Usually current admin
            'type' => 'edit_client',
            'reference_id' => $client->cl_id,
            'snapshot' => [
                'old' => $oldData,
                'new' => $client->toArray()
            ],
            'reg_by' => Auth::id(),
            'created_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => __('Client updated successfully')
        ]);
    }

    public function moveSavings(Request $request)
    {
        try {
            \Log::info('moveSavings hit', $request->all());
            $request->validate([
                'id_from' => 'required', // source member_no or old_member_no
                'id_to' => 'required', // target member_no or old_member_no
                'prj_ids' => 'required|array'
            ]);

            $mem_from = $request->id_from;
            $mem_to = $request->id_to;
            $units_input = $request->prj_ids;

            $clientTo = Client::where('cl_id', $mem_to)
                ->orWhere('member_no', $mem_to)
                ->orWhere('old_member_no', $mem_to)
                ->firstOrFail();

            $clientFrom = Client::where('cl_id', $mem_from)
                ->orWhere('member_no', $mem_from)
                ->orWhere('old_member_no', $mem_from)
                ->firstOrFail();

            DB::transaction(function () use ($clientFrom, $clientTo, $units_input) {
                $distinct_prj_ids = [];
                $transferred_details = [];

                // Define possible source IDs to match (Legacy inconsistency)
                $sourceIds = array_filter(array_unique([
                    $clientFrom->cl_id,
                    $clientFrom->member_no,
                    $clientFrom->old_member_no
                ]));

                \Log::info('Starting transfer transaction', [
                    'from_cl_id' => $clientFrom->cl_id,
                    'from_member' => $clientFrom->member_no,
                    'to_member' => $clientTo->member_no,
                    'source_search_ids' => $sourceIds,
                    'units_count' => count($units_input)
                ]);

                foreach ($units_input as $unitData) {
                    $unit = json_decode($unitData, true);
                    if (!$unit) {
                        // Fallback to treat it as a prj_id if it's not JSON
                        $prj_id = $unitData;
                        $distinct_prj_ids[] = $prj_id;

                        // 1. Audit/History for Reservations
                        $resHistorySql = "INSERT INTO mv_reservations SELECT * FROM default_reservations WHERE prj_id = ? AND cl_id IN (" . implode(',', array_fill(0, count($sourceIds), '?')) . ")";
                        $historyResRows = DB::insert($resHistorySql, array_merge([$prj_id], $sourceIds));

                        // 2. Update Reservations (Target cl_id = member_no per legacy)
                        $resRows = DB::table('default_reservations')
                            ->where('prj_id', $prj_id)
                            ->whereIn('cl_id', $sourceIds)
                            ->update(['cl_id' => $clientTo->member_no]);

                        \Log::info("Moved reservations for project $prj_id (Fallback)", ['rows' => $resRows, 'history' => $historyResRows]);

                        $transferred_details[] = "Project ID $prj_id";
                        continue;
                    }

                    $prj_id = $unit['prj_id'];
                    $bldg_id = $unit['bldg_id'];
                    $flat_num = $unit['flat_num'];
                    $distinct_prj_ids[] = $prj_id;

                    // 1. Audit/History for Reservations
                    $resDelHistorySql = "DELETE FROM mv_reservations WHERE prj_id = ? AND bldg_id = ? AND flat_num = ? AND cl_id IN (" . implode(',', array_fill(0, count($sourceIds), '?')) . ")";
                    DB::delete($resDelHistorySql, array_merge([$prj_id, $bldg_id, $flat_num], $sourceIds));

                    $resHistorySql = "INSERT INTO mv_reservations SELECT * FROM default_reservations WHERE prj_id = ? AND bldg_id = ? AND flat_num = ? AND cl_id IN (" . implode(',', array_fill(0, count($sourceIds), '?')) . ")";
                    $historyResRows = DB::insert($resHistorySql, array_merge([$prj_id, $bldg_id, $flat_num], $sourceIds));

                    // 2. Update Reservations (Target cl_id = member_no per legacy)
                    $resRows = DB::table('default_reservations')
                        ->where('prj_id', $prj_id)
                        ->where('bldg_id', $bldg_id)
                        ->where('flat_num', $flat_num)
                        ->whereIn('cl_id', $sourceIds)
                        ->update(['cl_id' => $clientTo->member_no]);

                    \Log::info("Moved reservations for unit $bldg_id/$flat_num (Project $prj_id)", ['rows' => $resRows, 'history' => $historyResRows]);

                    $transferred_details[] = "Unit $bldg_id/$flat_num (Project $prj_id)";
                }

                // Update Receipts per Project
                foreach (array_unique($distinct_prj_ids) as $prj_id) {
                    // 1. Audit/History for Receipts
                    // Get exact IDs of receipts to be moved to ensure clean history injection
                    $receiptIds = DB::table('default_receipts')
                        ->where('prj_id', $prj_id)
                        ->whereIn('cl_id', $sourceIds)
                        ->where('flag', 'Y')
                        ->pluck('id')
                        ->toArray();

                    if (!empty($receiptIds)) {
                        // Delete by ID to avoid PK collisions in mv_receipts
                        DB::table('mv_receipts')->whereIn('id', $receiptIds)->delete();

                        $recHistorySql = "INSERT INTO mv_receipts SELECT * FROM default_receipts WHERE id IN (" . implode(',', array_fill(0, count($receiptIds), '?')) . ")";
                        DB::insert($recHistorySql, $receiptIds);
                    }

                    // 2. Update Receipts (Target cl_id = cl_id per legacy)
                    $recRows = DB::table('default_receipts')
                        ->where('prj_id', $prj_id)
                        ->whereIn('cl_id', $sourceIds)
                        ->where('flag', 'Y')
                        ->update(['cl_id' => $clientTo->cl_id]);

                    \Log::info("Moved receipts for project $prj_id", ['rows' => $recRows, 'ids' => $receiptIds]);
                }

                // Update destination client status
                $clientTo->update(['move_status' => 'متنازل اليه']);

                // Log to HrAudit (Legacy Table)
                \App\Models\HrAudit::create([
                    'emp_id' => Auth::id(),
                    'type' => 'move_savings',
                    'reference_id' => $clientFrom->cl_id,
                    'snapshot' => [
                        'from_member' => $clientFrom->old_member_no,
                        'to_member' => $clientTo->old_member_no,
                        'transferred' => $transferred_details
                    ],
                    'reg_by' => Auth::id(),
                    'created_at' => now(),
                ]);

                // Log to AuditLog (New Table)
                \App\Models\AuditLog::create([
                    'action' => 'assignment_transfer',
                    'model' => 'Client',
                    'model_id' => $clientFrom->cl_id,
                    'details' => "Transferred savings from {$clientFrom->arabic_name} ({$clientFrom->old_member_no}) to {$clientTo->arabic_name} ({$clientTo->old_member_no}). Units: " . implode(', ', $transferred_details),
                    'user_id' => Auth::id(),
                    'ip_address' => request()->ip(),
                ]);
            });

            return response()->json([
                'success' => true,
                'message' => __('Savings and units moved successfully')
            ]);
        } catch (\Exception $e) {
            \Log::error('moveSavings failed: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        // Get client by cl_id or member_no
        $client = Client::where('cl_id', $id)->orWhere('member_no', $id)->firstOrFail();

        // Get all receipts for this client
        $receipts = DB::table('default_receipts as r')
            ->leftJoin('installment_type as t', 't.id', '=', 'r.type')
            ->leftJoin('banks as b', 'b.bank_id', '=', 'r.bank_account')
            ->where('r.cl_id', $client->cl_id)
            ->where('r.flag', 'Y')
            ->select('r.*', 't.type as type_txt', 'b.bank_name')
            ->orderByRaw("STR_TO_DATE(r.date ,'%d-%m-%Y'), STR_TO_DATE(r.date ,'%m-%d-%Y') ASC")
            ->get();

        // Calculate total
        $total = 0;
        foreach ($receipts as $receipt) {
            if ($receipt->chk != 'Y' && $receipt->prj_id != '99') {
                if (!in_array($receipt->type, ['100', '99', '111', '7', '8', '98'])) {
                    $total += $receipt->value;
                } else {
                    if ($receipt->type == '111') {
                        $total -= $receipt->value;
                    }
                }
            }
        }

        // Get client's reservation
        $reservation = DB::table('default_reservations as r')
            ->leftJoin('flats as f', function ($join) {
                $join->on('f.flt_id', '=', 'r.flt_id')
                    ->on('f.prj_id', '=', 'r.prj_id');
            })
            ->leftJoin('projects as p', 'p.prj_id', '=', 'r.prj_id')
            ->where('r.cl_id', $client->member_no)
            ->select('r.*', 'f.txt as flat_txt', 'p.name as project_name')
            ->first();

        // Get all projects for dropdown
        $projects = Project::where('flag', 'Y')->get();

        // Get all banks for dropdown
        $banks = Bank::all();

        // Get receipt types
        $receiptTypes = DB::table('installment_type')->get();

        // Get parking data if available
        $parking = collect();
        try {
            $tableExists = DB::select("SHOW TABLES LIKE 'default_park_resv'");
            if (!empty($tableExists)) {
                $parking = DB::table('default_park_resv')
                    ->where('cl_id', $client->member_no)
                    ->get();
            }
        } catch (\Exception $e) {
            \Log::info("Parking table not found: " . $e->getMessage());
        }

        // City and state arrays (from legacy)
        $cities = [
            '1' => ['القاهرة'],
            '2' => ['الجيزة'],
            '3' => ['الإسكندرية'],
            // Add more as needed
        ];

        $states = [
            '1' => ['مدينة نصر'],
            '2' => ['المعادي'],
            // Add more as needed
        ];

        $relations = [
            '1' => ['ابن'],
            '2' => ['أخ'],
            '3' => ['زوجة'],
            // Add more as needed
        ];

        $sessArr = [
            '1' => 'مرحله أولى',
            '11' => 'متبقيات مرحله اولى',
            '2' => 'مرحله ثانيه',
            '3' => 'مرحله ثالثه',
            '111' => 'بحرى'
        ];

        $subsess = '';
        if (isset($reservation->subsess) && $reservation->subsess == '1') {
            $subsess = 'متبقيات مرحله اولى';
        }

        // Check if there's updated data
        $cnt = 0;
        try {
            $tableExists = DB::select("SHOW TABLES LIKE 'default_clients_new'");
            if (!empty($tableExists)) {
                $cnt = DB::table('default_clients_new')->where('member_no', $client->member_no)->count();
            }
        } catch (\Exception $e) {
            \Log::info("default_clients_new table not found: " . $e->getMessage());
        }

        return view('admin.clients.show', compact(
            'client',
            'receipts',
            'total',
            'reservation',
            'projects',
            'banks',
            'receiptTypes',
            'parking',
            'cities',
            'states',
            'relations',
            'sessArr',
            'subsess',
            'cnt'
        ));
    }

    public function delayPay($id)
    {
        $client = Client::where('old_member_no', $id)->first();
        if (!$client) {
            return "Client not found";
        }

        $reservations = Reservation::where('cl_id', $client->member_no)
            ->with(['project', 'flat'])
            ->get();

        // Bulk fetch all receipts for this client once
        $allReceipts = Receipt::where('cl_id', $client->cl_id)
            ->where('flag', 'Y')
            ->orderBy('rev_date', 'asc')
            ->get();

        // Bulk fetch all schedules for projects/flats involved
        $projectIds = $reservations->pluck('prj_id')->unique();
        $flatIds = $reservations->pluck('flt_id')->unique();
        $allSchedules = TotalPrice::whereIn('prj_id', $projectIds)
            ->whereIn('flt_id', $flatIds)
            ->orderBy('id', 'asc')
            ->get();

        $reports = [];

        foreach ($reservations as $reservation) {
            $projectId = $reservation->prj_id;
            $flatId = $reservation->flt_id;

            // Filter from collections instead of querying in loop
            $receipts = $allReceipts->where('prj_id', $projectId);
            $schedule = $allSchedules->where('prj_id', $projectId)->where('flt_id', $flatId);

            $rows = [];
            $addedReceipts = [];

            foreach ($schedule as $installment) {
                $paidAmount = 0;
                $installmentDate = $installment->date;
                $row = [
                    'date' => $installmentDate,
                    'expected' => $installment->total,
                    'paid' => '',
                    'remaining' => ''
                ];

                foreach ($receipts as $receipt) {
                    // Logic from legacy: if(!in_array($row2->rev_date,$added))
                    // This logic seems slightly flawed in legacy or specific to their business rule. 
                    // It seems to try to match receipts to installments chronologically.
                    // If receipt date <= installment date.

                    if (!in_array($receipt->rev_date, $addedReceipts)) {
                        if ($receipt->rev_date <= $installmentDate) {
                            // Calculate cumulative paid up to this receipt's date
                            // Legacy: select sum(value) ... where rev_date <= $row2->rev_date
                            // This is expensive in a loop. I can optimize this in Laravel collection.

                            $currentPaid = $receipts->where('rev_date', '<=', $receipt->rev_date)->sum('value');

                            $remain = $installment->total - $currentPaid;
                            $remain = abs($remain); // Legacy does this

                            // Add a row for the receipt payment event
                            $rows[] = [
                                'date' => $receipt->rev_date,
                                'expected' => 'Paid',
                                'paid' => $currentPaid,
                                'remaining' => $remain
                            ];

                            $addedReceipts[] = $receipt->rev_date;
                        }
                    } else {
                        // Already handled this receipt date?
                        // Legacy logic is a bit spaghetti. 
                        // It iterates schedule, then inside iterates receipts. 
                        // If receipt already 'added' (by date?), it creates a different row?
                    }
                }

                // Add the installment row
                $rows[] = [
                    'date' => $installmentDate,
                    'expected' => $installment->total,
                    'paid' => '', // Legacy leaves this empty
                    'remaining' => ''
                ];
            }

            $reports[] = [
                'project_name' => $reservation->project->name ?? 'Unknown Project',
                'flat_name' => $reservation->flat->txt ?? 'Unknown Flat',
                'rows' => $rows
            ];
        }

        return view('admin.clients.delay_pay', compact('client', 'reports'));
    }

    public function viewComments($id = null)
    {
        if (!$id) {
            return redirect()->back()->with('error', 'Client ID is required');
        }

        $client = Client::where('old_member_no', $id)->first();
        if (!$client) {
            // Try finding by cl_id directly if old_member_no fails
            $client = Client::find($id);
            if (!$client)
                return "Client not found";
        }

        $comments = $client->comments()->orderBy('id', 'desc')->get();

        return view('admin.clients.comments', compact('client', 'comments'));
    }

    public function storeComment(Request $request)
    {
        $request->validate([
            'cl_id' => 'required',
            'comment' => 'required',
            'flag' => 'required',
        ]);

        $client = Client::find($request->cl_id); // Assuming cl_id is passed

        // Use authenticated admin ID
        $user = Auth::guard('admin')->id() ?? 0;

        $comment = new \App\Models\Comment();
        $comment->cl_id = $request->cl_id;
        $comment->mem_no = $client->file_no; // Legacy logic: $mem = $row1->file_no;
        $comment->comment = $request->comment;
        $comment->flag = $request->flag;
        $comment->user = $user;
        $comment->crdate = now();
        $comment->save();

        // Legacy: $this->add_log($data_c,"add",$tbl);
        // We'll skip logging for this pass or implement a basic Log model later.

        return redirect()->back()->with('success', 'Comment added successfully');
    }

    public function saveGarden(Request $request)
    {
        $request->validate([
            'prj_id' => 'required',
            'area' => 'required',
            'bldg_no' => 'required',
            'flat_num' => 'required',
            'mon' => 'required',
            'yr' => 'required',
            'gard_inst' => 'required',
        ]);

        $mon = $request->mon;
        if (strlen($mon) == 1) {
            $mon = '0' . $mon;
        }

        $data = [
            'year' => $request->yr,
            'month' => $mon,
            'gard_inst_amount' => $request->gard_inst,
            'flat_num' => $request->flat_num,
            'bldg_no' => $request->bldg_no,
            'area' => $request->area,
            'prj_id' => $request->prj_id,
            'total' => 0,
            'full_date' => "1-" . $mon . "-" . $request->yr,
            'yr_mon' => $request->yr . $mon
        ];

        \App\Models\GardenInstallment::create($data);

        // Legacy: $this->Add_log($data,"add",$tbl);

        return redirect()->back()->with('success', 'Garden installment saved successfully');
    }
    public function assignGarden(Request $request)
    {
        $request->validate([
            'garden_size' => 'required|numeric'
        ]);

        if ($request->has('resv_id')) {
            $reservation = \App\Models\Reservation::find($request->resv_id);
        } else {
            $request->validate([
                'cl_id' => 'required',
                'prj_id' => 'required',
                'bldg' => 'required',
                'flat_num' => 'required'
            ]);

            $reservation = \App\Models\Reservation::where('cl_id', $request->cl_id)
                ->where('prj_id', $request->prj_id)
                ->where('bldg_id', $request->bldg)
                ->where('flat_num', $request->flat_num)
                ->first();
        }

        if (!$reservation) {
            return redirect()->back()->with('error', 'Reservation not found for this unit');
        }

        $oldSize = $reservation->garden_size;
        $reservation->update([
            'garden' => 'Y',
            'garden_size' => $request->garden_size
        ]);

        // Log the assignment using the helper
        $this->saveAudit(null, 'assign_garden', $reservation->resv_id, [
            'old_size' => $oldSize,
            'new_size' => $request->garden_size,
            'prj_id' => $reservation->prj_id,
            'bldg_id' => $reservation->bldg_id,
            'flat_num' => $reservation->flat_num
        ]);

        return redirect()->back()->with('success', 'Garden assignment updated successfully');
    }

    public function deleteGarden(Request $request)
    {
        \Illuminate\Support\Facades\Log::info('deleteGarden called', $request->all());

        try {
            $request->validate([
                'resv_id' => 'required|exists:default_reservations,resv_id'
            ]);

            $reservation = \App\Models\Reservation::find($request->resv_id);
            if (!$reservation) {
                \Illuminate\Support\Facades\Log::error('Reservation not found for deleteGarden', ['resv_id' => $request->resv_id]);
                return response()->json(['error' => 'Reservation not found'], 404);
            }

            $this->saveAudit(null, 'delete_garden', $reservation->resv_id, [
                'garden_size' => $reservation->garden_size,
                'prj_id' => $reservation->prj_id,
                'bldg_id' => $reservation->bldg_id,
                'flat_num' => $reservation->flat_num
            ]);

            $reservation->update([
                'garden' => 'N',
                'garden_size' => 0
            ]);

            return redirect()->back()->with('success', 'Garden deleted successfully');
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('deleteGarden error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error deleting garden: ' . $e->getMessage());
        }
    }

    public function updateParking(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:default_park_resv,id',
            'bldg' => 'required',
            'park_no' => 'required'
        ]);

        $park = DB::table('default_park_resv')->where('id', $request->id)->first();

        DB::table('default_park_resv')->where('id', $request->id)->update([
            'bldg' => $request->bldg,
            'park_no' => $request->park_no
        ]);

        $this->saveAudit(null, 'update_parking', $request->id, [
            'old_bldg' => $park->bldg,
            'old_park_no' => $park->park_no,
            'new_bldg' => $request->bldg,
            'new_park_no' => $request->park_no
        ]);

        return redirect()->back()->with('success', 'Parking assignment updated successfully');
    }

    public function createGarden()
    {
        $projects = Project::all();
        $installments = \App\Models\GardenInstallment::orderBy('id', 'desc')->paginate(50);
        return view('admin.clients.garden', compact('projects', 'installments'));
    }

    public function saveParking(Request $request)
    {
        $request->validate([
            'prj_id' => 'required',
            'bldg_no' => 'required',
            'park_no' => 'required',
            'mon' => 'required',
            'yr' => 'required',
            'park_inst' => 'required',
        ]);

        $mon = $request->mon;
        if (strlen($mon) == 1) {
            $mon = '0' . $mon;
        }

        $data = [
            'yr' => $request->yr,
            'mon' => $mon,
            'inst_amount' => $request->park_inst,
            'park_no' => $request->park_no,
            'bldg_no' => $request->bldg_no,
            'prj_id' => $request->prj_id,
            'date' => "1-" . $mon . "-" . $request->yr,
            'yr_mon' => $request->yr . $mon
        ];

        \App\Models\ParkingInstallment::create($data);

        return redirect()->back()->with('success', 'Parking installment saved successfully');
    }

    public function refund(Request $request)
    {
        $request->validate([
            'cl_id' => 'required', // This seems to be Member No in legacy "Msabry::Refund", but let's verify usage. 
            // Legacy: $qr = "select * from default_clients where member_no = '".$_POST['cl_id']."'";
            // So input 'cl_id' is actually 'member_no'.
            'prj_id' => 'required',
            'flt_id' => 'required',
            'recs' => 'required', // Pipe separated IDs
        ]);

        $member_no = $request->cl_id;
        $client = Client::where('member_no', $member_no)->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found'], 404);
        }

        $real_cl_id = $client->cl_id;
        $recs_str = rtrim($request->recs, '|');
        $rec_ids = explode('|', $recs_str);

        // Calculate Amount
        $amount = \App\Models\Receipt::where('cl_id', $real_cl_id)
            ->whereIn('id', $rec_ids)
            ->where('flag', 'Y')
            ->sum('value');

        \App\Models\Pending::create([
            'cl_id' => $member_no, // Legacy stores member_no in cl_id field of pendings?
            // Legacy: $data_p = array('cl_id'=>$_POST['cl_id']...
            // Yes, legacy stores the INPUT cl_id (which is member_no) into the table.
            'prj_id' => $request->prj_id,
            'flt_id' => $request->flt_id,
            'amount' => $amount,
            'type' => 'BK',
            'recs' => $recs_str
        ]);

        return response()->json(['success' => 'Refund request created successfully', 'amount' => $amount]);
    }

    public function processPending(Request $request)
    {
        $id = $request->id;
        $pending = \App\Models\Pending::find($id);

        if (!$pending) {
            return response()->json(['error' => 'Pending request not found'], 404);
        }

        if ($pending->status == 'Y') {
            return response()->json(['error' => 'Request already processed'], 400);
        }

        DB::transaction(function () use ($pending) {
            $cl_id = $pending->cl_id; // Member No (Input)
            $prj_id = $pending->prj_id;
            // Get internal Client ID
            $client = Client::where('member_no', $cl_id)->first();
            $internal_cl_id = $client->cl_id;

            if ($pending->type == 'MV') {
                $cl_to_member_no = $pending->cl_to;
                $targetClient = Client::where('member_no', $cl_to_member_no)->first();
                $target_internal_cl_id = $targetClient->cl_id;

                // Archive Receipts
                $receipts = \App\Models\Receipt::where('cl_id', $internal_cl_id)->get();
                foreach ($receipts as $receipt) {
                    \App\Models\MvReceipt::create((array) $receipt->getAttributes());
                }

                // Archive Reservations
                $reservations = \App\Models\Reservation::where('cl_id', $cl_id)->get();
                foreach ($reservations as $res) {
                    \App\Models\MvReservation::create((array) $res->getAttributes());
                }

                // Transfer Data
                \App\Models\Receipt::where('cl_id', $internal_cl_id)->update(['cl_id' => $target_internal_cl_id]);
                \App\Models\Reservation::where('cl_id', $cl_id)->update(['cl_id' => $cl_to_member_no]);

            } else { // refund / BK
                $recs_str = $pending->recs;
                $rec_ids = explode('|', rtrim($recs_str, '|'));

                // Archive to Cancel Reservations
                $reservations = \App\Models\Reservation::where('cl_id', $cl_id)->get();
                foreach ($reservations as $res) {
                    \App\Models\CancelReservation::create((array) $res->getAttributes());
                }

                // Delete Reservation
                \App\Models\Reservation::where('cl_id', $cl_id)->delete();

                // Refund Transaction in Accounts
                // Soft delete current active account
                \App\Models\Account::where('cl_id', $internal_cl_id)->update(['flag' => 'N']);

                // Add Refund Record
                \App\Models\Account::create([
                    'cl_id' => $internal_cl_id,
                    'amount' => $pending->amount,
                    'crdate' => now()->format('d-m-Y h:i:sa'),
                    'flag' => 'Y' // New Active Balance? Or just a record? Logic implies returning money.
                ]);

                // Soft Delete Receipts
                \App\Models\Receipt::whereIn('id', $rec_ids)->update([
                    'flag' => 'N',
                    'deleted_at' => now(),
                    'deleted_by' => Auth::guard('admin')->id() ?? 1
                ]);
            }

            // Audit Log: Process Pending Request
            $this->saveAudit($cl_id, $pending->type == 'MV' ? 'process_transfer_request' : 'cash_refund', $pending->id, [
                'pending_id' => $pending->id,
                'type' => $pending->type,
                'amount' => $pending->amount,
                'prj_id' => $pending->prj_id,
                'status' => 'Processed'
            ]);

            // Mark Pending as Processed
            $pending->status = 'Y';
            $pending->save();
        });

        return response()->json(['success' => 'Request processed successfully']);
    }

    public function setupTables()
    {
        DB::statement('CREATE TABLE IF NOT EXISTS mv_receipts LIKE default_receipts');
        DB::statement('CREATE TABLE IF NOT EXISTS mv_reservations LIKE default_reservations');
        DB::statement('CREATE TABLE IF NOT EXISTS default_cancel_res LIKE default_reservations');
        return "Tables created";
    }

    public function saveTransaction(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric',
            'mem_no' => 'required', // Target Old Member No
            'from_id' => 'required', // Source Client ID (cl_id)
            'base_amount' => 'required|numeric',
            'prj_id' => 'required',
            'rec_ids' => 'required',
            'type' => 'required',
        ]);

        $amount = $request->amount;
        $mem_no = $request->mem_no;
        $from_id = $request->from_id;
        $base_amount = $request->base_amount;
        $prj_id = $request->prj_id;
        $rec_ids = explode('|', rtrim($request->rec_ids, '|'));
        $type = $request->type;
        $file_no = $request->file_no;

        // Find Target Client
        $targetClientQuery = \App\Models\Client::where('old_member_no', $mem_no);
        if ($file_no) {
            $targetClientQuery->where('member_no', $file_no);
        }
        $targetClient = $targetClientQuery->first();

        if (!$targetClient) {
            return redirect()->back()->with('error', 'Target client not found.');
        }

        $to_id = $targetClient->cl_id;
        $trans_by = Auth::guard('admin')->id() ?? 1; // Fallback to 1 for robust testing if needed

        DB::transaction(function () use ($from_id, $to_id, $base_amount, $amount, $trans_by, $prj_id, $rec_ids, $type, $request) {
            $now = now();
            $rec_id_timestamp = $now->timestamp;

            // 1. Update Source Account (Soft Delete current active)
            \App\Models\Account::where('cl_id', $from_id)->update(['flag' => 'N']);

            // 2. Add New Source Account Balance
            \App\Models\Account::create([
                'cl_id' => $from_id,
                'amount' => ($base_amount - $amount),
                'crdate' => $now->format('d-m-Y h:i:sa'),
                'receipt_id' => $rec_id_timestamp,
                'trans_date' => $now->format('d-m-Y h:i:sa'),
                'trans_by' => $trans_by,
                'flag' => 'Y'
            ]);

            // 3. Add Target Account Record
            \App\Models\Account::create([
                'cl_id' => $to_id,
                'amount' => $amount,
                'crdate' => $now->format('d-m-Y h:i:sa'),
                'prj_id' => $prj_id,
                'receipt_id' => $rec_id_timestamp,
                'amount_from' => $from_id,
                'trans_date' => $now->format('d-m-Y h:i:sa'),
                'trans_by' => $trans_by,
                'flag' => 'Y'
            ]);

            // Audit Log
            $this->saveAudit($from_id, 'transfer_funds', $rec_id_timestamp, [
                'from_cl_id' => $from_id,
                'to_cl_id' => $to_id,
                'amount' => $amount,
                'base_amount' => $base_amount,
                'type' => $type,
                'prj_id' => $prj_id
            ]);

            // 4. Transform Receipts
            $originalReceipts = \App\Models\Receipt::whereIn('id', $rec_ids)->get();

            foreach ($originalReceipts as $receipt) {
                if ($type == 'to_cl') {
                    $receipt->update(['flag' => 'N']);

                    // Create Copy for Target
                    \App\Models\Receipt::create([
                        'number' => $receipt->number,
                        'bnk_number' => $receipt->bnk_number,
                        'cl_id' => $to_id,
                        'reg_by' => $trans_by,
                        'value' => $receipt->value,
                        'date' => $receipt->date,
                        'type' => $receipt->type,
                        'prj_id' => $prj_id,
                        'created_at' => $now,
                        'trans_id' => $rec_id_timestamp,
                        'bank_account' => $receipt->bank_account,
                        'flag' => 'Y'
                    ]);

                } else {
                    $receipt->update(['chk' => 'Y']);
                }
            }

            if ($type != 'to_cl') {
                // Create Single Check Receipt
                \App\Models\Receipt::create([
                    'number' => $request->chk_num,
                    'chk_ids' => implode('-', $rec_ids),
                    'cl_id' => $to_id,
                    'reg_by' => $trans_by,
                    'value' => $amount,
                    'date' => $request->chk_date,
                    'type' => '111', // Check Type
                    'prj_id' => $prj_id,
                    'created_at' => $now,
                    'trans_id' => $rec_id_timestamp,
                    'chk' => 'Y',
                    'bank_account' => $request->chk_bnk,
                    'flag' => 'Y'
                ]);
            }
        });

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => __('Transaction saved successfully')
            ]);
        }

        return redirect()->back()->with('success', 'Transaction saved successfully.');
    }

    public function toggleReview(Request $request)
    {
        $request->validate([
            'id' => 'required|integer',
            'stat' => 'required|in:Y,N'
        ]);

        DB::table('default_receipts')
            ->where('id', $request->id)
            ->update(['rev' => $request->stat]);

        return response()->json(['success' => true]);
    }

    // Receipt Management Methods

    public function addReceipt(Request $request, $id)
    {
        $request->validate([
            'number' => 'required',
            'bnk_number' => 'required',
            'value' => 'required|numeric',
            'date' => 'required',
            'prj_id' => 'required',
            'type' => 'required',
            'bank_account' => 'required'
        ]);

        // Get current user
        $currentUser = Auth::guard('admin')->user()->username ?? Auth::user()->username ?? 'unknown';

        // Date validation for non-admin users
        $adminUsers = ['admin', 'mayoussef', 'mtantawy17', 'ziad_mahmoud', 'raafat.moustafa', 'moustafa.mohamed', 'ahmed_zedan'];
        if (!in_array($currentUser, $adminUsers)) {
            $dateStr = $request->date;
            if (strpos($dateStr, '-') !== false) {
                $parts = explode('-', $dateStr);
                $year = $parts[2] ?? date('Y');
                if ($year != date('Y')) {
                    return response()->json(['error' => 'غير مسموح بتسجيل ايصالات قديمه'], 400);
                }
            }
        }

        // Check for duplicate receipt number
        $duplicateReceipt = DB::table('default_receipts')
            ->where('number', $request->number)
            ->where('flag', 'Y')
            ->count();

        if ($duplicateReceipt > 0) {
            return response()->json(['error' => 'رقم الايصال موجود من قبل, يرجى التاكد من الرقم الصحيح'], 400);
        }

        // Check for duplicate bank number
        $duplicateBank = DB::table('default_receipts')
            ->where('bnk_number', $request->bnk_number)
            ->where('flag', 'Y')
            ->count();

        if ($duplicateBank > 0) {
            return response()->json(['error' => 'رقم الايصال البنكى موجود من قبل, يرجى التاكد من الرقم الصحيح'], 400);
        }

        // Check if receipt number equals bank number
        if ($request->number == $request->bnk_number) {
            return response()->json(['error' => 'رقم الحركه او الحساب البنكى غير صحيح'], 400);
        }

        // Create receipt
        $receiptId = DB::table('default_receipts')->insertGetId([
            'cl_id' => $id,
            'number' => $request->number,
            'bnk_number' => $request->bnk_number,
            'value' => $request->value,
            'date' => $request->date,
            'prj_id' => $request->prj_id,
            'type' => $request->type,
            'bank_account' => $request->bank_account,
            'reg_by' => Auth::guard('admin')->id() ?? Auth::id() ?? 1,
            'created_at' => now(),
            'flag' => 'Y'
        ]);

        // Handle serial number tracking if starts with 5555
        if (substr($request->number, 0, 4) == '5555') {
            DB::table('serial_temp')->insert(['serial' => $request->number]);
        }

        // Audit Log
        $this->saveAudit(null, 'add_receipt', $receiptId, [
            'cl_id' => $id,
            'number' => $request->number,
            'value' => $request->value,
            'date' => $request->date
        ]);

        return response()->json(['success' => true, 'id' => $receiptId]);
    }

    public function updateReceipt(Request $request, $id)
    {
        $request->validate([
            'number' => 'required',
            'bnk_number' => 'required',
            'value' => 'required|numeric',
            'date' => 'required',
            'prj_id' => 'required',
            'type' => 'required',
            'bank_account' => 'required'
        ]);

        // Check for duplicate receipt number (excluding current)
        $duplicateReceipt = DB::table('default_receipts')
            ->where('number', $request->number)
            ->where('id', '!=', $id)
            ->where('flag', 'Y')
            ->count();

        if ($duplicateReceipt > 0) {
            return response()->json(['error' => 'رقم الايصال موجود من قبل, يرجى التاكد من الرقم الصحيح'], 400);
        }

        // Check for duplicate bank number (excluding current)
        $duplicateBank = DB::table('default_receipts')
            ->where('bnk_number', $request->bnk_number)
            ->where('id', '!=', $id)
            ->where('flag', 'Y')
            ->count();

        if ($duplicateBank > 0) {
            return response()->json(['error' => 'رقم الايصال البنكى موجود من قبل, يرجى التاكد من الرقم الصحيح'], 400);
        }

        // Update receipt
        DB::table('default_receipts')
            ->where('id', $id)
            ->update([
                'number' => $request->number,
                'bnk_number' => $request->bnk_number,
                'value' => $request->value,
                'date' => $request->date,
                'prj_id' => $request->prj_id,
                'type' => $request->type,
                'bank_account' => $request->bank_account,
                'chk' => $request->type == '999' ? 'Y' : 'N'
            ]);

        // Audit Log
        $this->saveAudit(null, 'edit_receipt', $id, [
            'number' => $request->number,
            'value' => $request->value,
            'date' => $request->date,
            'new_bnk_number' => $request->bnk_number
        ]);

        return response()->json(['success' => true]);
    }

    public function deleteReceipt($id)
    {
        DB::table('default_receipts')
            ->where('id', $id)
            ->update(['flag' => 'N']);

        // Audit Log
        $this->saveAudit(null, 'delete_receipt', $id, ['reason' => 'soft delete']);

        return response()->json(['success' => true]);
    }

    public function bulkDeleteReceipts(Request $request)
    {
        $ids = $request->ids;
        if (empty($ids)) {
            return response()->json(['error' => 'No receipts selected'], 400);
        }

        DB::table('default_receipts')
            ->whereIn('id', $ids)
            ->update(['flag' => 'N']);

        return response()->json(['success' => true]);
    }

    public function checkReceiptNumber(Request $request)
    {
        $number = $request->number;
        $id = $request->id ?? null;
        $field = $request->field ?? 'number'; // Default to 'number', can be 'bnk_number'

        $query = DB::table('default_receipts')
            ->where($field, $number)
            ->where('flag', 'Y');

        if ($id) {
            $query->where('id', '!=', $id);
        }

        $count = $query->count();

        return response()->json(['count' => $count]);
    }

    public function generateReceiptSerial()
    {
        // Get the last serial number starting with 5555
        $lastSerial = DB::table('default_receipts')
            ->where('number', 'like', '5555%')
            ->orderBy('number', 'desc')
            ->value('number');

        if ($lastSerial) {
            $nextSerial = (int) $lastSerial + 1;
        } else {
            $nextSerial = 55550001;
        }

        // Store in temp table
        DB::table('serial_temp')->insert(['serial' => $nextSerial]);

        return response()->json(['serial' => $nextSerial]);
    }

    public function generateBankSerial()
    {
        // Get the last bank serial number
        $lastSerial = DB::table('default_receipts')
            ->where('bnk_number', '!=', '0')
            ->where('bnk_number', '!=', '')
            ->whereNotNull('bnk_number')
            ->orderBy('bnk_number', 'desc')
            ->value('bnk_number');

        if ($lastSerial && is_numeric($lastSerial)) {
            $nextSerial = (int) $lastSerial + 1;
        } else {
            $nextSerial = 1;
        }

        return response()->json(['serial' => $nextSerial]);
    }

    public function lockReceipt(Request $request, $id)
    {
        $status = $request->status; // Y or N

        DB::table('default_receipts')
            ->where('id', $id)
            ->update(['locked' => $status]);

        return response()->json(['success' => true]);
    }

    public function printCheck(Request $request)
    {
        $value = $request->value;
        $date = $request->date;
        $name = $request->name;
        $bank_id = $request->bank_id;

        $bank = Bank::where('bank_id', $bank_id)->first();
        $bank_name = $bank ? $bank->bank_name : '';

        // Simple manual tafqit or placeholder
        $tafqit = $value . ' جنيه مصري فقط لا غير';

        return view('admin.clients.print_check', compact('value', 'date', 'name', 'bank_name', 'tafqit'));
    }

    // Reservation Management Methods

    public function getReservations($id)
    {
        $client = Client::where('cl_id', $id)
            ->orWhere('member_no', $id)
            ->orWhere('old_member_no', $id)
            ->firstOrFail();

        $reservations = DB::table('default_reservations as r')
            ->leftJoin('flats as f', function ($join) {
                $join->on('f.flt_id', '=', 'r.flt_id')
                    ->on('f.prj_id', '=', 'r.prj_id');
            })
            ->leftJoin('projects as p', 'p.prj_id', '=', 'r.prj_id')
            ->where(function ($query) use ($client) {
                $query->where('r.cl_id', $client->member_no)
                    ->orWhere('r.cl_id', $client->old_member_no)
                    ->orWhere('r.cl_id', $client->cl_id);
            })
            ->select('r.*', 'f.txt as flat_txt', 'p.name as project_name')
            ->get();

        return response()->json(['reservations' => $reservations]);
    }

    public function addReservation(Request $request, $id)
    {
        $request->validate([
            'prj_id' => 'required',
            'bldg_id' => 'required',
            'flat_num' => 'required',
            'floor' => 'required'
        ]);

        $client = Client::where('cl_id', $id)->orWhere('member_no', $id)->firstOrFail();

        // Check if unit is already reserved
        $existing = DB::table('default_reservations')
            ->where('prj_id', $request->prj_id)
            ->where('bldg_id', $request->bldg_id)
            ->where('flat_num', $request->flat_num)
            ->count();

        if ($existing > 0) {
            return response()->json(['error' => 'هذه الوحده مخصصه من قبل'], 400);
        }

        // Find flt_id from flats table
        $flat = DB::table('flats')
            ->where('prj_id', $request->prj_id)
            ->where('bldg_id', $request->bldg_id)
            ->where('txt', $request->flat_num)
            ->first();

        $flt_id = $flat ? $flat->flt_id : 0;

        DB::table('default_reservations')->insert([
            'cl_id' => $client->member_no,
            'prj_id' => $request->prj_id,
            'flt_id' => $flt_id,
            'bldg_id' => $request->bldg_id,
            'flat_num' => $request->flat_num,
            'floor' => $request->floor,
            'resv_date' => $request->resv_date ?? now()->format('d-m-Y h:i:sa'),
            'resv_by' => Auth::guard('admin')->id() ?? Auth::id() ?? 1
        ]);

        // Audit Log
        $this->saveAudit($client->member_no, 'add_reservation', 0, [
            'prj_id' => $request->prj_id,
            'bldg_id' => $request->bldg_id,
            'flat_num' => $request->flat_num,
            'floor' => $request->floor
        ]);

        return response()->json(['success' => true]);
    }

    public function deleteReservation(Request $request)
    {
        $deleted = DB::table('default_reservations')
            ->where('prj_id', $request->prj_id)
            ->where('bldg_id', $request->bldg_id)
            ->where('flat_num', $request->flat_num)
            ->delete();

        if ($deleted) {
            // Audit Log
            $this->saveAudit(null, 'delete_reservation', 0, [
                'prj_id' => $request->prj_id,
                'bldg_id' => $request->bldg_id,
                'flat_num' => $request->flat_num
            ]);

            return response()->json(['success' => true]);
        }

        return response()->json(['error' => 'Reservation not found'], 404);
    }

    // Review Status Methods

    public function setReviewStatus(Request $request, $id)
    {
        $status = $request->status; // Y or N

        DB::table('default_clients')
            ->where('cl_id', $id)
            ->update(['rev' => $status]);

        return response()->json(['success' => true]);
    }

    public function viewLiveData($memberNo)
    {
        // Get updated client data from new table
        $client = DB::table('default_clients_new')
            ->where('member_no', $memberNo)
            ->first();

        if (!$client) {
            return "No updated data found";
        }

        return view('admin.clients.live_data', compact('client'));
    }

    public function viewOldData($memberNo)
    {
        // Get original client data
        $client = Client::where('member_no', $memberNo)->firstOrFail();

        return view('admin.clients.old_data', compact('client'));
    }

    public function getReceiptData(Request $request)
    {
        $clId = $request->cl_id;
        $prjId = $request->prj_id;

        $receipts = DB::table('default_receipts')
            ->where('cl_id', $clId)
            ->where('prj_id', $prjId)
            ->where('flag', 'Y')
            ->where('chk', 'N')
            ->get();

        return response()->json(['receipts' => $receipts]);
    }

    public function deleteParking($id)
    {
        \Illuminate\Support\Facades\Log::info('deleteParking called', ['id' => $id]);
        try {
            $park = DB::table('default_park_resv')->where('id', $id)->first();
            if ($park) {
                $this->saveAudit(null, 'delete_park', $id, (array) $park);
                DB::table('default_park_resv')->where('id', $id)->delete();
            } else {
                \Illuminate\Support\Facades\Log::warning('deleteParking: Park not found', ['id' => $id]);
                return response()->json(['error' => 'Parking not found'], 404);
            }
            return redirect()->back()->with('success', 'Parking deleted successfully');
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('deleteParking error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error deleting parking: ' . $e->getMessage());
        }
    }
    public function checkAvailability(Request $request)
    {
        $prj_id = $request->prj_id;
        $bldg_id = $request->bldg_id;
        $floor = $request->floor;

        $query = DB::table('flats')
            ->where('prj_id', $prj_id)
            ->where('bldg_id', $bldg_id);

        if ($floor) {
            $query->where('floor', $floor);
        }

        // Get flats that are NOT in reservations (available)
        // Or flag checks if flats table has status
        // Usually flats have 'status' or checks against reservations table

        // Let's assume we return all flats matching criteria AND their reservation status
        // Or just available ones.
        // Legacy system check:
        // usually checks if (prj_id, flt_id) exists in default_reservations

        $flats = $query->select('flt_id', 'txt', 'bldg_id', 'floor', 'flat_num')->get();

        $availableFlats = [];
        foreach ($flats as $flat) {
            $isReserved = DB::table('default_reservations')
                ->where('prj_id', $prj_id)
                ->where('flt_id', $flat->flt_id)
                ->exists();

            if (!$isReserved) {
                $availableFlats[] = $flat;
            }
        }

        return response()->json(['flats' => $availableFlats]);
    }

    public function getProjects()
    {
        $projects = Project::where('flag', 'Y')->select('prj_id', 'name')->get();
        return response()->json($projects); // Return array directly
    }

    public function getBuildings(Request $request)
    {
        $prj_id = $request->prj_id;
        $buildings = DB::table('flats')
            ->where('prj_id', $prj_id)
            ->select('bldg_id')
            ->distinct()
            ->orderBy('bldg_id')
            ->get();

        return response()->json(['buildings' => $buildings]);
    }
    private function saveAudit($empId, $type, $refId, $snapshot)
    {
        \App\Models\HrAudit::create([
            'emp_id' => $empId, // Can be null or client ID if relevant
            'type' => $type,
            'reference_id' => $refId,
            'snapshot' => $snapshot, // Will be cast to json array automatically by model
            'reg_by' => Auth::guard('admin')->id() ?? Auth::id() ?? 1
        ]);
    }
}
