<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class MigrateUploadedFiles extends Command
{
    protected $signature = 'files:migrate';
    protected $description = 'Migrate uploaded files from legacy_code to storage and update database paths';

    public function handle()
    {
        $this->info('Starting file migration...');

        // Get all uploaded files from database
        $files = DB::table('default_rec_files')
            ->where('flag', '1')
            ->where('moved', '0')
            ->get();

        $this->info("Found {$files->count()} files to migrate.");

        $migrated = 0;
        $failed = 0;
        $skipped = 0;

        foreach ($files as $file) {
            // Try legacy_code location first
            $legacyPath = base_path('legacy_code/' . $file->path);

            // If not found, try without legacy_code prefix
            if (!file_exists($legacyPath)) {
                $legacyPath = base_path($file->path);
            }

            // Check if file exists in legacy location
            if (!file_exists($legacyPath)) {
                $this->warn("File not found: {$file->path}");
                $skipped++;
                continue;
            }

            // Create new storage path
            $newPath = 'uploads/receipts/' . $file->user_id . '/' . $file->filename;
            $storagePath = storage_path('app/public/' . $newPath);

            // Create directory if it doesn't exist
            $directory = dirname($storagePath);
            if (!file_exists($directory)) {
                mkdir($directory, 0755, true);
            }

            // Copy file to new location
            if (copy($legacyPath, $storagePath)) {
                // Update database with new path
                DB::table('default_rec_files')
                    ->where('id', $file->id)
                    ->update([
                        'path' => 'storage/' . $newPath
                    ]);

                $migrated++;
                $this->info("✓ Migrated: {$file->filename}");
            } else {
                $failed++;
                $this->error("✗ Failed to copy: {$file->filename}");
            }
        }

        $this->newLine();
        $this->info("Migration complete!");
        $this->info("Migrated: {$migrated}");
        $this->warn("Skipped: {$skipped}");
        $this->error("Failed: {$failed}");

        if ($migrated > 0) {
            $this->newLine();
            $this->info("Don't forget to create storage link if not exists:");
            $this->comment("php artisan storage:link");
        }

        return 0;
    }
}
